﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Windows;
using System.Windows.Controls;
using System.Windows.Data;
using System.Windows.Documents;
using System.Windows.Input;
using System.Windows.Media;
using System.Windows.Media.Imaging;
using System.Windows.Navigation;
using System.Windows.Shapes;
using System.ComponentModel;

namespace Oscilloscope
{
    /// <summary>
    /// Follow steps 1a or 1b and then 2 to use this custom control in a XAML file.
    ///
    /// Step 1a) Using this custom control in a XAML file that exists in the current project.
    /// Add this XmlNamespace attribute to the root element of the markup file where it is 
    /// to be used:
    ///
    ///     xmlns:MyNamespace="clr-namespace:Oscilloscope"
    ///
    ///
    /// Step 1b) Using this custom control in a XAML file that exists in a different project.
    /// Add this XmlNamespace attribute to the root element of the markup file where it is 
    /// to be used:
    ///
    ///     xmlns:MyNamespace="clr-namespace:Oscilloscope;assembly=Oscilloscope"
    ///
    /// You will also need to add a project reference from the project where the XAML file lives
    /// to this project and Rebuild to avoid compilation errors:
    ///
    ///     Right click on the target project in the Solution Explorer and
    ///     "Add Reference"->"Projects"->[Browse to and select this project]
    ///
    ///
    /// Step 2)
    /// Go ahead and use your control in the XAML file.
    ///
    ///     <MyNamespace:OscilloscopeControl/>
    ///
    /// </summary>
    /// 
    [Description("A quick plot control that will scroll data point by point.")]
    public class OscilloscopeControl : Control
    {
        [Description("Minimum displayed data value that will not be considered clipped."), Category("Common Properties")]
        public double YMin
        {
            get { return ymin; }
            set { if (value < ymax) ymin = value; else throw new ArgumentOutOfRangeException("YMin"); }
        }

        [Description("Maximum displayed data value that will not be considered clipped."), Category("Common Properties")]
        public double YMax
        {
            get { return ymax; }
            set { if (value > ymin) ymax = value; else throw new ArgumentOutOfRangeException("YMax"); }
        }

        public int XPixRange { get { return pixel_w; } }
        public int YPixRange { get { return pixel_h; } }

        public bool AutoScale
        {
            get { return autoscale; }
            set { autoscale = value; }
        }

        public Color DataColor
        {
            get { return datacolor; }
            set { datacolor = value; updateColors(true); plotBitmap.AddDirtyRect(fullRect); }
        }

        public Color ClippedColor
        {
            get { return clippedcolor; }
            set { clippedcolor = value; updateColors(true); plotBitmap.AddDirtyRect(fullRect); }
        }

        int pixel_w = 0, pixel_h = 0;
        double ymin=0, ymax=1;
        bool autoscale = false;
        Color datacolor = Color.FromRgb(0, 0, 0);
        Color clippedcolor = Color.FromRgb(255, 0, 0);
        int[] pixelData;
        WriteableBitmap plotBitmap;
        Int32Rect fullRect;
        Image plotImg;

        static OscilloscopeControl()
        {
            DefaultStyleKeyProperty.OverrideMetadata(typeof(OscilloscopeControl), new FrameworkPropertyMetadata(typeof(OscilloscopeControl)));
        }

        public override void OnApplyTemplate()
        {
            plotImg = (Image)this.Template.FindName("PART_OSCPLOT", this);

            pixel_w = (int)(plotImg.Width);
            pixel_h = (int)(plotImg.Height);

            pixelData = new int[pixel_w];
            plotBitmap = new WriteableBitmap(pixel_w, pixel_h, 96, 96, PixelFormats.Bgra32, null);
            fullRect = new Int32Rect(0, 0, pixel_w, pixel_h);

            plotImg.Source = plotBitmap;

            clearPlot_do();
        }
        
        public void PlotPoint(double data)
        {
            this.Dispatcher.Invoke(new Action(() => { plotPoint_do(data); }));
        }

        void plotPoint_do(double data)
        {
            plotBitmap.Lock();
            plotData(true);
            for (int a = 1; a < pixel_w; a++)
            {
                pixelData[a - 1] = pixelData[a];
            }
            pixelData[pixel_w - 1] = convertData(data);
            plotData(false);

            plotBitmap.AddDirtyRect(fullRect);
            plotBitmap.Unlock();
        }

        public void PlotPoints(double[] data)
        {
            this.Dispatcher.Invoke(new Action(() => { plotPoints_do(data); }));
        }

        void plotPoints_do(double[] data)
        {
            int dl = data.Count();
            int ds = 0;
            if (dl > pixel_w)
                ds = dl - pixel_w;
            int pixy;

            plotBitmap.Lock();
            plotData(true);

            for (int a = (pixel_w - (dl - ds)); a < pixel_w; a++)
            {
                pixy = convertData(data[ds]);
                ds++;
                pixelData[a] = pixy;
            }

            plotData(false);
            plotBitmap.AddDirtyRect(fullRect);
            plotBitmap.Unlock();
        }

        public void ClearPlot()
        {
            this.Dispatcher.Invoke(new Action(() => { clearPlot_do(); }));
        }

        void clearPlot_do()
        {
            plotBitmap.Lock();
            updateColors(false);
            for (int a = 0; a < pixel_w; a++)
            {
                pixelData[a] = pixel_h-1;
            }
            plotBitmap.AddDirtyRect(fullRect);
            plotBitmap.Unlock();
        }

        int convertData(double data)
        {
            int pixy;
            if (autoscale)
            {
                if (data > ymax)
                {
                    ymax = data;
                    pixy = 0;
                }
                else
                {
                    if (data < ymin)
                    {
                        ymin = data;
                        pixy = pixel_h - 1;
                    }
                    else
                        pixy = 1 + ((int)((ymax - data) / (ymax - ymin) * (pixel_h - 3)));
                }
            }
            else
            {
                if (data > ymax)
                    pixy = 0;
                else
                    if (data < ymin)
                        pixy = pixel_h - 1;
                    else
                        pixy = 1 + ((int)((ymax - data) / (ymax - ymin) * (pixel_h - 3)));
            }

            return pixy;
        }

        void plotData(bool erase)
        {
            int p_adr;

            unsafe
            {
                byte* pbuff = (byte*)plotBitmap.BackBuffer.ToPointer();
                for (int a = 0; a < pixel_w; a++)
                {
                    p_adr = (pixelData[a] * pixel_w + a) * 4;
                    if(erase)
                        pbuff[p_adr + 3] = 0;
                    else
                        pbuff[p_adr + 3] = 255;
                }
            }
        }

        void updateColors(bool keepTransparency)
        {
            int p_adr;
            unsafe
            {
                byte* pbuff = (byte*)plotBitmap.BackBuffer.ToPointer();

                for (int b = 0; b < pixel_w; b++)
                {
                    p_adr = (b) * 4;
                    pbuff[p_adr] = clippedcolor.B;
                    pbuff[p_adr + 1] = clippedcolor.G;
                    pbuff[p_adr + 2] = clippedcolor.R;
                    if (!keepTransparency)
                        pbuff[p_adr + 3] = 0;
                }

                for (int a = 1; a < pixel_h-1; a++)
                {
                    for (int b = 0; b < pixel_w; b++)
                    {
                        p_adr = (a * pixel_w + b) * 4;
                        pbuff[p_adr] = datacolor.B;
                        pbuff[p_adr + 1] = datacolor.G;
                        pbuff[p_adr + 2] = datacolor.R;
                        if(!keepTransparency)
                            pbuff[p_adr + 3] = 0;
                    }
                }

                for (int b = 0; b < pixel_w; b++)
                {
                    p_adr = ((pixel_h - 1) * pixel_w + b) * 4;
                    pbuff[p_adr] = clippedcolor.B;
                    pbuff[p_adr + 1] = clippedcolor.G;
                    pbuff[p_adr + 2] = clippedcolor.R;
                    if (!keepTransparency)
                        pbuff[p_adr + 3] = 0;
                }
            }
        }
    }
}
