﻿using System;
using System.Windows;
using System.Windows.Controls;
using System.Windows.Media;
using System.Windows.Shapes;

namespace Photogrammetry
{
    /// <summary>
    /// Interaction logic for Plane2ProjControl.xaml
    /// </summary>
    public partial class Plane2ProjControl: ProjControlBase
    {
        ProjectedPlane M2p = null;
        string activeTool = null;
     
        public Plane2ProjControl(Canvas mainCanvas) : base(mainCanvas)
        {
            InitializeComponent();
        }

        private void ToolButton_Checked(object sender, RoutedEventArgs e)
        {
            RadioButton snd = (RadioButton)sender;
            SetTool((string)snd.Content);
            e.Handled = true;
        }

        private void Scale_TextChanged(object sender, TextChangedEventArgs e)
        {
            runComputation();
        }

        public override void SetTool(string toolName)
        {
            if (double.IsNaN(myCanvas.Width))
            {
                return;
            }
            activeTool = toolName;
            if (M2p == null)
            {
                M2p = new ProjectedPlane
                {
                    axis1 = new ProjectedAxis(),
                    axis2 = new ProjectedAxis()
                };
            }
            switch (toolName)
            {
                case "X vp1":
                    if (M2p.axis1.primary == null)
                    {
                        M2p.axis1.primary = MakeLine(XAxisBrush, true);
                    }
                    break;
                case "X vp2":
                    if (M2p.axis1.secondary == null)
                    {
                        M2p.axis1.secondary = MakeLine(XAxisBrush, true);
                    }
                    break;
                case "Y vp1":
                    if (M2p.axis2.primary == null)
                    {
                        M2p.axis2.primary = MakeLine(YAxisBrush, true);
                    }
                    break;
                case "Y vp2":
                    if (M2p.axis2.secondary == null)
                    {
                        M2p.axis2.secondary = MakeLine(YAxisBrush, true);
                    }
                    break;
                case "X ref":
                    if (M2p.a1ref1 == null)
                    {
                        M2p.a1ref1 = MakeLine(XAxisBrush);
                        M2p.a1ref1.Tag = new Point(0, 0);
                        M2p.a1ref2 = MakeLine(XAxisBrush);
                        M2p.a1ref2.Tag = new Point(myCanvas.Width - 1, myCanvas.Height - 1);
                    }
                    break;
                case "Y ref":
                    if (M2p.a2ref1 == null)
                    {
                        M2p.a2ref1 = MakeLine(YAxisBrush);
                        M2p.a2ref1.Tag = new Point(0, 0);
                        M2p.a2ref2 = MakeLine(YAxisBrush);
                        M2p.a2ref2.Tag = new Point(myCanvas.Width - 1, myCanvas.Height - 1);
                    }
                    break;
                case "Meas":
                    if (M2p.measLine == null)
                    {
                        M2p.measLine = MakeLine(MeasLineBrush);
                    }
                    break;
                default:
                    activeTool = null;
                    break;
            }
        }

        public override Point GetSelectionPt(bool rightClick)
        {
            if (activeTool == null || myCanvas == null)
            {
                return new Point();
            }
            if (M2p != null)
            {
                switch (activeTool)
                {
                    case "X vp1":
                        if (M2p.axis1.primary != null)
                        {
                            return GetLineEnd(M2p.axis1.primary, !rightClick);
                        }
                        break;
                    case "X vp2":
                        if (M2p.axis1.secondary != null)
                        {
                            return GetLineEnd(M2p.axis1.secondary, !rightClick);
                        }
                        break;
                    case "Y vp1":
                        if (M2p.axis2.primary != null)
                        {
                            return GetLineEnd(M2p.axis2.primary, !rightClick);
                        }
                        break;
                    case "Y vp2":
                        if (M2p.axis2.secondary != null)
                        {
                            return GetLineEnd(M2p.axis2.secondary, !rightClick);
                        }
                        break;
                    case "X ref":
                        if (!rightClick && M2p.a1ref1 != null)
                        {
                            return (Point)M2p.a1ref1.Tag;
                        }
                        if (rightClick && M2p.a1ref2 != null)
                        {
                            return (Point)M2p.a1ref2.Tag;
                        }
                        break;
                    case "Y ref":
                        if (!rightClick && M2p.a2ref1 != null)
                        {
                            return (Point)M2p.a2ref1.Tag;
                        }
                        if (rightClick && M2p.a2ref2 != null)
                        {
                            return (Point)M2p.a2ref2.Tag;
                        }
                        break;
                    case "Meas":
                        if (M2p.measLine != null)
                        {
                            return GetLineEnd(M2p.measLine, !rightClick);
                        }
                        break;
                }
            }
            return new Point();
        }

        private void UserControl_IsVisibleChanged(object sender, DependencyPropertyChangedEventArgs e)
        {
            Show((bool)e.NewValue);
            if((bool)e.NewValue == true)
            {
                //reset tool and update event handler
            }
        }

        private void runComputation()
        {
            if (!IsInitialized)
            {
                return;
            }
            
            if (M2p == null || !M2p.IsReady())
            {
                M2resx.Text = "dx";
                M2resy.Text = "dy";
                M2resl.Text = "L";
                M2resa.Text = "A";
                return;
            }

            double refdx, refdy;
            if(!double.TryParse(M2RefX.Text, out refdx))
            {
                refdx = double.NaN;
            }
            if(!double.TryParse(M2RefY.Text, out refdy))
            {
                refdy = double.NaN;
            }

            //complete the intersections of the reference lines to get a rectangle
            ParametricLine pl1 = new ParametricLine(M2p.axis1.primary); //x vp1 reference
            ParametricLine pl2 = new ParametricLine(M2p.axis2.primary); //y vp1 reference
            Point origin = pl1.Intersect(pl2);
            ParametricLine pl3 = new ParametricLine(M2p.axis1.secondary); //x vanishing point reference
            VanishingPoint xvp = new VanishingPoint(pl1, pl3);
            ParametricLine pl4 = new ParametricLine(M2p.axis2.secondary); //y vanishing point reference
            VanishingPoint yvp = new VanishingPoint(pl2, pl4);

            //project the reference points onto corresponding axes
            Point rx1, rx2, ry1, ry2;
            rx1 = yvp.ProjectTo(pl1, (Point)M2p.a1ref1.Tag);
            rx2 = yvp.ProjectTo(pl1, (Point)M2p.a1ref2.Tag);
            ry1 = xvp.ProjectTo(pl2, (Point)M2p.a2ref1.Tag);
            ry2 = xvp.ProjectTo(pl2, (Point)M2p.a2ref2.Tag);

            //project the query points onto corresponding axes
            Point mmp1 = GetLineEnd(M2p.measLine, true);
            Point mmp2 = GetLineEnd(M2p.measLine, false);
            Point mx1, mx2, my1, my2;
            mx1 = yvp.ProjectTo(pl1, mmp1);
            mx2 = yvp.ProjectTo(pl1, mmp2);
            my1 = xvp.ProjectTo(pl2, mmp1);
            my2 = xvp.ProjectTo(pl2, mmp2);

            Point mlpx, mlpy; //measurement line points
            Point mlmx1, mlmx2, mlmy1, mlmy2, mlrx1, mlrx2, mlry1, mlry2;
            //Measurement line projected measurement and reference points

            ParametricLine horizonLine = VanishingPoint.GetHorizonLine(xvp, yvp);
            ParametricLine imageLine = new ParametricLine(horizonLine, origin);
            if (!xvp.IsPoint)
            {
                if (!yvp.IsPoint)
                {
                    //horizonLine and imageLine unused
                    //both xvp and yvp very far away
                    //for x and y axis the initial projection is already expected to be linear
                    mlmx1 = mx1;
                    mlmx2 = mx2;
                    mlmy1 = my1;
                    mlmy2 = my2;
                    mlrx1 = rx1;
                    mlrx2 = rx2;
                    mlry1 = ry1;
                    mlry2 = ry2;
                }
                else
                {
                    mlpy = findMeasurementPoint(horizonLine, pl2, origin); //y-axis measurement point
                                                                           //now we can project with references
                                                                           //for x-axis the initial projection is already expected to be linear
                    mlmx1 = mx1;
                    mlmx2 = mx2;
                    mlmy1 = imageLine.ProjectOntoFromPoint(mlpy, my1);
                    mlmy2 = imageLine.ProjectOntoFromPoint(mlpy, my2);
                    mlrx1 = rx1;
                    mlrx2 = rx2;
                    mlry1 = imageLine.ProjectOntoFromPoint(mlpy, ry1);
                    mlry2 = imageLine.ProjectOntoFromPoint(mlpy, ry2);
                }
            }
            else
            {
                if (!yvp.IsPoint)
                {
                    //yvp very far away
                    mlpx = findMeasurementPoint(horizonLine, pl1, origin); //x-axis measurement point
                                                                           //now we can project with references
                                                                           //for y-axis the initial projection is already expected to be linear
                    mlmx1 = imageLine.ProjectOntoFromPoint(mlpx, mx1);
                    mlmx2 = imageLine.ProjectOntoFromPoint(mlpx, mx2);
                    mlmy1 = my1;
                    mlmy2 = my2;
                    mlrx1 = imageLine.ProjectOntoFromPoint(mlpx, rx1);
                    mlrx2 = imageLine.ProjectOntoFromPoint(mlpx, rx2);
                    mlry1 = ry1;
                    mlry2 = ry2;
                }
                else
                {
                    //both vanishing points valid
                    mlpx = findMeasurementPoint(horizonLine, pl1, origin); //x-axis measurement point
                    mlpy = findMeasurementPoint(horizonLine, pl2, origin); //y-axis measurement point
                                                                           //now we can project with references
                    mlmx1 = imageLine.ProjectOntoFromPoint(mlpx, mx1);
                    mlmx2 = imageLine.ProjectOntoFromPoint(mlpx, mx2);
                    mlmy1 = imageLine.ProjectOntoFromPoint(mlpy, my1);
                    mlmy2 = imageLine.ProjectOntoFromPoint(mlpy, my2);
                    mlrx1 = imageLine.ProjectOntoFromPoint(mlpx, rx1);
                    mlrx2 = imageLine.ProjectOntoFromPoint(mlpx, rx2);
                    mlry1 = imageLine.ProjectOntoFromPoint(mlpy, ry1);
                    mlry2 = imageLine.ProjectOntoFromPoint(mlpy, ry2);
                }
            }

            ParametricLine puxline = new ParametricLine(mlmx1, mlmx2);
            ParametricLine puyline = new ParametricLine(mlmy1, mlmy2);
            ParametricLine prxline = new ParametricLine(mlrx1, mlrx2);
            ParametricLine pryline = new ParametricLine(mlry1, mlry2);
            double rxlength = prxline.NormDist;
            double rylength = pryline.NormDist;
            double mxlength = puxline.NormDist;
            double mylength = puyline.NormDist;
            double dx = (mxlength / rxlength) * refdx;
            if (prxline.DotProduct(puxline) < 0)
            {
                //opposite sense
                dx = -dx;
            }
            double dy = (mylength / rylength) * refdy;
            if (pryline.DotProduct(puyline) < 0)
            {
                //opposite sense
                dy = -dy;
            }
            double length = Math.Sqrt(dx * dx + dy * dy);
            double angle = Math.Atan2(dy, dx) / Math.PI * 180;

            M2resx.Text = "dx : " + dx.ToString("G4") + " <xlen>";
            M2resy.Text = "dy : " + dy.ToString("G4") + " <ylen>";
            M2resl.Text = "L  : " + length.ToString("G4") + " <len>";
            M2resa.Text = "A  : " + angle.ToString("G4") + " [deg]";
        }

        public override void ApplyTool(Point selectPt, bool rightClick)
        {
            if (M2p != null)
            {
                switch (activeTool)
                {
                    case "X vp1":
                        if (M2p.axis1.primary != null)
                        {
                            AdjustLine(M2p.axis1.primary, !rightClick, selectPt);
                        }
                        if (M2p.a2ref1 != null)
                        {
                            adjustTickLine(M2p.a2ref1, M2p.GetLineThrough((Point)M2p.a2ref1.Tag, true));
                        }
                        if (M2p.a2ref2 != null)
                        {
                            adjustTickLine(M2p.a2ref2, M2p.GetLineThrough((Point)M2p.a2ref2.Tag, true));
                        }
                        break;
                    case "X vp2":
                        if (M2p.axis1.secondary != null)
                        {
                            AdjustLine(M2p.axis1.secondary, !rightClick, selectPt);
                        }
                        if (M2p.a2ref1 != null)
                        {
                            adjustTickLine(M2p.a2ref1, M2p.GetLineThrough((Point)M2p.a2ref1.Tag, true));
                        }
                        if (M2p.a2ref2 != null)
                        {
                            adjustTickLine(M2p.a2ref2, M2p.GetLineThrough((Point)M2p.a2ref2.Tag, true));
                        }
                        break;
                    case "Y vp1":
                        if (M2p.axis2.primary != null)
                        {
                            AdjustLine(M2p.axis2.primary, !rightClick, selectPt);
                        }
                        if (M2p.a1ref1 != null)
                        {
                            adjustTickLine(M2p.a1ref1, M2p.GetLineThrough((Point)M2p.a1ref1.Tag, false));
                        }
                        if (M2p.a1ref2 != null)
                        {
                            adjustTickLine(M2p.a1ref2, M2p.GetLineThrough((Point)M2p.a1ref2.Tag, false));
                        }
                        break;
                    case "Y vp2":
                        if (M2p.axis2.secondary != null)
                        {
                            AdjustLine(M2p.axis2.secondary, !rightClick, selectPt);
                        }
                        if (M2p.a1ref1 != null)
                        {
                            adjustTickLine(M2p.a1ref1, M2p.GetLineThrough((Point)M2p.a1ref1.Tag, false));
                        }
                        if (M2p.a1ref2 != null)
                        {
                            adjustTickLine(M2p.a1ref2, M2p.GetLineThrough((Point)M2p.a1ref2.Tag, false));
                        }
                        break;
                    case "X ref":
                        if (!rightClick && M2p.a1ref1 != null)
                        {
                            adjustTickLine(M2p.a1ref1, M2p.GetLineThrough(selectPt, false));
                        }
                        if (rightClick && M2p.a1ref2 != null)
                        {
                            adjustTickLine(M2p.a1ref2, M2p.GetLineThrough(selectPt, false));
                        }
                        break;
                    case "Y ref":
                        if (!rightClick && M2p.a2ref1 != null)
                        {
                            adjustTickLine(M2p.a2ref1, M2p.GetLineThrough(selectPt, true));
                        }
                        if (rightClick && M2p.a2ref2 != null)
                        {
                            adjustTickLine(M2p.a2ref2, M2p.GetLineThrough(selectPt, true));
                        }
                        break;
                    case "Meas":
                        if (M2p.measLine != null)
                        {
                            AdjustLine(M2p.measLine, !rightClick, selectPt);
                        }
                        break;
                }
                runComputation();
            }
        }

        static Point findMeasurementPoint(ParametricLine horizonLine, ParametricLine vanishingLine, Point origin)
        {
            ParametricLine mplx = findMeasurementSlope(horizonLine, vanishingLine);
            return horizonLine.ProjectOntoFromSlope(mplx, origin); //measurement point
        }

        static ParametricLine findMeasurementSlope(ParametricLine horizonLine, ParametricLine vanishingLine)
        {
            //approximate best-resolution achieved at projected angle half that of vanishing point and perpendicular to it
            double va = horizonLine.AngleTo(vanishingLine);
            double v2;
            if (Math.Abs(va) > (Math.PI / 2))
            {
                v2 = Math.PI - Math.Abs(va);
                if (Math.Sign(va) > 0)
                {
                    v2 = Math.PI / 2 - v2 / 2;
                }
                else
                {
                    v2 = -(Math.PI / 2 - v2 / 2);
                }
            }
            else
            {
                v2 = Math.Abs(va);
                if (Math.Sign(va) > 0)
                {
                    v2 = Math.PI / 2 + v2 / 2;
                }
                else
                {
                    v2 = -(Math.PI / 2 + v2 / 2);
                }
            }
            return horizonLine.Rotate(v2); //imageLine.Rotate is easier to interpret but they are parallel and only slope is used below
        }

        void adjustTickLine(Line lrm, ParametricLine pl)
        {
            Point xl1 = pl.Traverse(10);
            Point xl2 = pl.Traverse(-10);
            lrm.X1 = xl1.X;
            lrm.Y1 = xl1.Y;
            lrm.X2 = xl2.X;
            lrm.Y2 = xl2.Y;
            lrm.Tag = pl.Offset;
        }
    }
}
