﻿using System;
using System.Windows;
using System.Windows.Controls;
using System.Windows.Media;

namespace Photogrammetry
{
    /// <summary>
    /// Interaction logic for Plane1ProjControl.xaml
    /// </summary>
    public partial class Plane1ProjControl : ProjControlBase
    {
        ProjectedPlane M2p = null;
        string activeTool = null;

        public Plane1ProjControl(Canvas mainCanvas) : base(mainCanvas)
        {
            InitializeComponent();
        }

        private void ToolButton_Checked(object sender, RoutedEventArgs e)
        {
            RadioButton snd = (RadioButton)sender;
            SetTool((string)snd.Content);
            e.Handled = true;
        }

        private void Scale_TextChanged(object sender, TextChangedEventArgs e)
        {
            runComputation();
        }

        public override void SetTool(string toolName)
        {
            if (double.IsNaN(myCanvas.Width))
            {
                return;
            }
            activeTool = toolName;
            if (M2p == null)
            {
                M2p = new ProjectedPlane
                {
                    axis1 = new ProjectedAxis(),
                    axis2 = new ProjectedAxis()
                };
            }
            switch (toolName)
            {
                case "X vp1":
                    if (M2p.axis1.primary == null)
                    {
                        M2p.axis1.primary = MakeLine(XAxisBrush, true);
                    }
                    break;
                case "X vp2":
                    if (M2p.axis1.secondary == null)
                    {
                        M2p.axis1.secondary = MakeLine(XAxisBrush, true);
                    }
                    break;
                case "Y vp1":
                    if (M2p.axis2.primary == null)
                    {
                        M2p.axis2.primary = MakeLine(YAxisBrush, true);
                    }
                    break;
                case "Y vp2":
                    if (M2p.axis2.secondary == null)
                    {
                        M2p.axis2.secondary = MakeLine(YAxisBrush, true);
                    }
                    break;
                case "Ref":
                    if (M2p.a1ref1 == null)
                    {
                        M2p.a1ref1 = MakeLine(RefLineBrush);
                    }
                    break;
                case "Meas":
                    if (M2p.measLine == null)
                    {
                        M2p.measLine = MakeLine(MeasLineBrush);
                    }
                    break;
                default:
                    activeTool = null;
                    break;
            }
        }

        public override void ApplyTool(Point selectPt, bool rightClick)
        {
            if (M2p != null)
            {
                switch (activeTool)
                {
                    case "X vp1":
                        if (M2p.axis1.primary != null)
                        {
                            AdjustLine(M2p.axis1.primary, !rightClick, selectPt);
                        }
                        break;
                    case "X vp2":
                        if (M2p.axis1.secondary != null)
                        {
                            AdjustLine(M2p.axis1.secondary, !rightClick, selectPt);
                        }
                        break;
                    case "Y vp1":
                        if (M2p.axis2.primary != null)
                        {
                            AdjustLine(M2p.axis2.primary, !rightClick, selectPt);
                        }
                        break;
                    case "Y vp2":
                        if (M2p.axis2.secondary != null)
                        {
                            AdjustLine(M2p.axis2.secondary, !rightClick, selectPt);
                        }
                        break;
                    case "Ref":
                        if (M2p.a1ref1 != null)
                        {
                            AdjustLine(M2p.a1ref1, !rightClick, selectPt);
                        }
                        break;
                    case "Meas":
                        if (M2p.measLine != null)
                        {
                            AdjustLine(M2p.measLine, !rightClick, selectPt);
                        }
                        break;
                }
                runComputation();
            }
        }

        public override Point GetSelectionPt(bool rightClick)
        {
            if (activeTool == null || myCanvas == null)
            {
                return new Point();
            }
            if (M2p != null)
            {
                switch (activeTool)
                {
                    case "X vp1":
                        if (M2p.axis1.primary != null)
                        {
                            return GetLineEnd(M2p.axis1.primary, !rightClick);
                        }
                        break;
                    case "X vp2":
                        if (M2p.axis1.secondary != null)
                        {
                            return GetLineEnd(M2p.axis1.secondary, !rightClick);
                        }
                        break;
                    case "Y vp1":
                        if (M2p.axis2.primary != null)
                        {
                            return GetLineEnd(M2p.axis2.primary, !rightClick);
                        }
                        break;
                    case "Y vp2":
                        if (M2p.axis2.secondary != null)
                        {
                            return GetLineEnd(M2p.axis2.secondary, !rightClick);
                        }
                        break;
                    case "Ref":
                        if (M2p.a1ref1 != null)
                        {
                            return GetLineEnd(M2p.a1ref1, !rightClick);
                        }
                        break;
                    case "Meas":
                        if (M2p.measLine != null)
                        {
                            return GetLineEnd(M2p.measLine, !rightClick);
                        }
                        break;
                }
            }
            return new Point();
        }

        private void UserControl_IsVisibleChanged(object sender, DependencyPropertyChangedEventArgs e)
        {
            Show((bool)e.NewValue);
            if((bool)e.NewValue == true)
            {
                //reset tool and update event handler
            }
        }

        private void runComputation()
        {
            if (!IsInitialized)
            {
                return;
            }

            if (M2p == null || !M2p.axis1.IsReady() || !M2p.axis2.IsReady() || M2p.a1ref1 == null || M2p.measLine == null)
            {
                M2aresl.Text = "L";
                M2aresa.Text = "A";
                M2arest.Text = "T";
                M2note.Text = "Not ready";
                return;
            }

            double reflen, afov_in;
            if (!double.TryParse(M2aRef.Text, out reflen))
            {
                reflen = double.NaN;
            }
            if (!double.TryParse(M2aFOV.Text, out afov_in))
            {
                afov_in = double.NaN;
            }

            //complete the intersections of the reference lines to get vanishing points
            ParametricLine pl1 = new ParametricLine(M2p.axis1.primary); //x vp1 reference
            ParametricLine pl2 = new ParametricLine(M2p.axis2.primary); //y vp1 reference
            ParametricLine pl3 = new ParametricLine(M2p.axis1.secondary); //x vanishing point reference
            VanishingPoint xvp = new VanishingPoint(pl1, pl3);
            ParametricLine pl4 = new ParametricLine(M2p.axis2.secondary); //y vanishing point reference
            VanishingPoint yvp = new VanishingPoint(pl2, pl4);

            ParametricLine horizon = VanishingPoint.GetHorizonLine(xvp, yvp);
            Point img_origin = new Point(myCanvas.Width / 2, myCanvas.Height / 2);
            ParametricLine meas_line;

            Point rx1, rx2, ry1, ry2;
            Point ref1 = GetLineEnd(M2p.a1ref1, true);
            Point ref2 = GetLineEnd(M2p.a1ref1, false);

            Point mx1, mx2, my1, my2;
            Point mmp1 = GetLineEnd(M2p.measLine, true);
            Point mmp2 = GetLineEnd(M2p.measLine, false);

            double phi, theta, di, afov_out;

            if ((!xvp.IsPoint) && (!yvp.IsPoint))
            {
                //both VPs are far away
                //we can then project onto the horizon line (which is average of slopes) from the two VPs
                meas_line = new ParametricLine(horizon, img_origin);
                rx1 = xvp.ProjectTo(meas_line, ref1);
                rx2 = xvp.ProjectTo(meas_line, ref2);
                ry1 = yvp.ProjectTo(meas_line, ref1);
                ry2 = yvp.ProjectTo(meas_line, ref2);
                mx1 = xvp.ProjectTo(meas_line, mmp1);
                mx2 = xvp.ProjectTo(meas_line, mmp2);
                my1 = yvp.ProjectTo(meas_line, mmp1);
                my2 = yvp.ProjectTo(meas_line, mmp2);
                theta = pl2.AngleTo(horizon);
                phi = 0;
                afov_out = double.NaN;
                M2note.Text = "Using parallel projection";
            }
            else
            {
                //both VPs nearby
                ParametricLine virtualx = new ParametricLine(horizon.Perpendicular(), img_origin);
                Point intercept = virtualx.Intersect(horizon);
                ParametricLine vxtr = new ParametricLine(intercept, img_origin);
                Point dvp1, dvp2;

                theta = 0;

                if ((!xvp.IsPoint) || (!yvp.IsPoint))
                {
                    if (double.IsNaN(afov_in))
                    {
                        M2aresl.Text = "L";
                        M2aresa.Text = "A";
                        M2arest.Text = "T";
                        M2note.Text = "VP too far - enter AFOV";
                        return;
                    }
                    else
                    {
                        if (!yvp.IsPoint)
                        {
                            theta = 0;
                        }
                        else
                        {
                            theta = Math.PI / 2;
                        }
                    }
                }
                //need to calculate AFOV
                ParametricLine e1 = new ParametricLine(intercept, xvp.GetPoint());
                ParametricLine e2 = new ParametricLine(intercept, yvp.GetPoint());
                double f = Math.Sqrt(e1.NormDist * e2.NormDist);
                Point viewpt = vxtr.Traverse(f); //from this point the angle to both VPs will be 90 degrees

                double cosp = vxtr.NormDist / f;
                phi = Math.Acos(cosp);
                di = vxtr.NormDist * Math.Tan(phi);
                afov_out = Math.Atan2((Math.Max(myCanvas.Width, myCanvas.Height) / 2), di) * 2;

                if (xvp.IsPoint && yvp.IsPoint)
                {
                    ParametricLine ovpl1 = new ParametricLine(viewpt, xvp.GetPoint());
                    theta = ovpl1.AngleTo(vxtr);
                }

                if (!double.IsNaN(afov_in))
                {
                    //we have AFOV as input
                    double vx = ParametricLine.GetDistance(img_origin, intercept);
                    double fov = afov_in / 180 * Math.PI;
                    di = (Math.Max(myCanvas.Width, myCanvas.Height) / 2) / Math.Tan(fov / 2);
                    phi = Math.Atan2(di, vx);
                    double sp = di / Math.Sin(phi);
                    ParametricLine measuring = new ParametricLine(horizon, intercept);
                    dvp1 = measuring.Traverse(sp);
                    dvp2 = measuring.Traverse(-sp);
                    M2note.Text = "Using AFOV as input (angles may be inaccurate)";
                }
                else
                {
                    //use calculated AFOV
                    ParametricLine dvpl1 = new ParametricLine(vxtr.Rotate(Math.PI / 4), viewpt);
                    ParametricLine dvpl2 = new ParametricLine(vxtr.Rotate(-Math.PI / 4), viewpt);

                    dvp1 = dvpl1.Intersect(horizon);
                    dvp2 = dvpl2.Intersect(horizon);

                    if(Math.Abs(Math.IEEERemainder(theta / Math.PI * 180, 90)) < 5.1)
                    {
                        //theta is close to a multiple of 90 degrees meaning one of the vanishing points is much farther than the other
                        M2note.Text = "Accuracy error likely in calc. AFOV";
                    }
                    else
                    {
                        M2note.Text = "Using calculated viewpoint";
                    }
                }

                Point meas_offset = vxtr.Traverse(Math.Max(myCanvas.Width, myCanvas.Height) / 2);
                meas_line = new ParametricLine(horizon, meas_offset);
                //project the reference points onto corresponding axes


                rx1 = meas_line.ProjectOntoFromPoint(dvp1, ref1);
                ry1 = meas_line.ProjectOntoFromPoint(dvp2, ref1);
                rx2 = meas_line.ProjectOntoFromPoint(dvp1, ref2);
                ry2 = meas_line.ProjectOntoFromPoint(dvp2, ref2);

                mx1 = meas_line.ProjectOntoFromPoint(dvp1, mmp1);
                my1 = meas_line.ProjectOntoFromPoint(dvp2, mmp1);
                mx2 = meas_line.ProjectOntoFromPoint(dvp1, mmp2);
                my2 = meas_line.ProjectOntoFromPoint(dvp2, mmp2);
            }

            ParametricLine rpx = new ParametricLine(rx1, rx2);
            ParametricLine rpy = new ParametricLine(ry1, ry2);
            double rdx = rpx.DotProduct(meas_line) > 0 ? rpx.NormDist : (-rpx.NormDist);
            double rdy = rpy.DotProduct(meas_line) > 0 ? rpy.NormDist : (-rpy.NormDist);
            //ParametricLine rpc = new ParametricLine(rdx, rdy, 0, 0);
            
            
            ParametricLine mpx = new ParametricLine(mx1, mx2);
            ParametricLine mpy = new ParametricLine(my1, my2);
            double mdx = mpx.DotProduct(meas_line) > 0 ? mpx.NormDist : (-mpx.NormDist);
            double mdy = mpy.DotProduct(meas_line) > 0 ? mpy.NormDist : (-mpy.NormDist);
            ParametricLine ref_obj = new ParametricLine(rdx, rdy, 0, 0);
            ParametricLine query_obj = new ParametricLine(mdx, mdy, 0, 0);
            double rlen = (query_obj.NormDist / ref_obj.NormDist) * reflen;
            double rang = -ref_obj.AngleTo(query_obj) / Math.PI * 180; //[deg]
            
            M2aresl.Text = "L  : " + rlen.ToString("G4") + " <len>";
            M2aresa.Text = "A  : " + rang.ToString("G4") + " [deg]";
            theta = theta / Math.PI * 180;
            M2arest.Text = "theta  : " + theta.ToString("G4") + " [deg]";
            phi = phi / Math.PI * 180;
            M2aresp.Text = "phi  : " + phi.ToString("G4") + " [deg]";
            afov_out = afov_out / Math.PI * 180;
            M2aresf.Text = "AFOV  : " + afov_out.ToString("G4") + " [deg]";
        }
    }
}
