﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Windows;
using System.Windows.Media;
using System.Windows.Shapes;

namespace Photogrammetry
{
    public class ParametricLine
    {
        private double slopeX, slopeY, offsetX, offsetY;
        private double normd = 0;
        public double NormDist { get { return normd; } }
        public Point Offset { get { return new Point(offsetX, offsetY); } }

        private ParametricLine()
        {

        }

        public ParametricLine(double SlopeX, double SlopeY, double OffsetX, double OffsetY)
        {
            slopeX = SlopeX;
            slopeY = SlopeY;

            Normalize();

            offsetX = OffsetX;
            offsetY = OffsetY;
        }

        private void Normalize()
        {
            normd = Math.Sqrt(slopeX * slopeX + slopeY * slopeY);
            slopeX /= normd;
            slopeY /= normd;
        }

        public ParametricLine(Line fromLine)
        {
            slopeX = fromLine.X2 - fromLine.X1;
            slopeY = fromLine.Y2 - fromLine.Y1;

            Normalize();

            offsetX = fromLine.X1;
            offsetY = fromLine.Y1;
        }

        public ParametricLine(ParametricLine slope, ParametricLine offset)
        {
            slopeX = slope.slopeX;
            slopeY = slope.slopeY;
            normd = slope.normd;

            offsetX = offset.offsetX;
            offsetY = offset.offsetY;
        }

        public ParametricLine(ParametricLine slope, Point offset)
        {
            slopeX = slope.slopeX;
            slopeY = slope.slopeY;
            normd = slope.normd;

            offsetX = offset.X;
            offsetY = offset.Y;
        }

        public ParametricLine(Point pt1, Point pt2)
        {
            offsetX = pt1.X;
            offsetY = pt1.Y;

            slopeX = pt2.X - pt1.X;
            slopeY = pt2.Y - pt1.Y;

            Normalize();
        }

        public Point Traverse(double t)
        {
            return new Point(offsetX + t * slopeX, offsetY + t * slopeY);
        }

        public Point Intersect(ParametricLine l2)
        {
            //solve a matrix equation for parametric lines as slope1*t1+offset1 = slope2*t2+offset2
            double a = slopeX;
            double b = -l2.slopeX;
            double c = slopeY;
            double d = -l2.slopeY;
            //double dd = a * d - b * c; //with parallel lines this will approach zero
            double x1 = l2.offsetX - offsetX;
            double x2 = l2.offsetY - offsetY;
            double t1 = ((x1 * d) - (x2 * b)) / (a * d - b * c); //parameter for line 1
            //double t2 = (-(x1 * c) + (x2 * a)) / (a * d - b * c);
            return new Point(t1 * slopeX + offsetX, t1 * slopeY + offsetY);
        }

        public Point ProjectOntoFromPoint(Point pjsource, Point pjobject)
        {
            ParametricLine pjl = new ParametricLine(pjsource, pjobject);
            return this.Intersect(pjl);
        }

        public Point ProjectOntoFromSlope(ParametricLine pjslope, Point pjobject)
        {
            ParametricLine pjl = new ParametricLine(pjslope, pjobject);
            return this.Intersect(pjl);
        }

        public ParametricLine AverageSlope(ParametricLine bsl, bool positive)
        {
            ParametricLine pjl = new ParametricLine();
            if (positive)
            {
                pjl.slopeX = slopeX + bsl.slopeX;
                pjl.slopeY = slopeY + bsl.slopeY;
            }
            else
            {
                pjl.slopeX = slopeX - bsl.slopeX;
                pjl.slopeY = slopeY - bsl.slopeY;
            }
            pjl.Normalize();

            pjl.offsetX = 0;
            pjl.offsetY = 0;

            return pjl;
        }

        public double AngleTo(ParametricLine pl2)
        {
            double dp = slopeX * pl2.slopeX + slopeY * pl2.slopeY;
            double det = slopeX * pl2.slopeY - slopeY * pl2.slopeX;

            return Math.Atan2(det,dp);
        }

        public double DotProduct(ParametricLine pl2)
        {
            return slopeX * pl2.slopeX + slopeY * pl2.slopeY;
        }

        public ParametricLine Perpendicular(bool ccw = true)
        {
            ParametricLine pl = new ParametricLine();
            if (ccw)
            {
                pl.slopeX = -slopeY;
                pl.slopeY = slopeX;
            }
            else
            {
                pl.slopeX = slopeY;
                pl.slopeY = -slopeX;
            }
            pl.offsetX = offsetX;
            pl.offsetY = offsetY;
            pl.normd = normd;
            return pl;
        }

        public ParametricLine Rotate(double angle)
        {
            ParametricLine pl = new ParametricLine();
            pl.offsetX = offsetX;
            pl.offsetY = offsetY;
            double c = Math.Cos(angle);
            double s = Math.Sin(angle);
            pl.slopeX = c * slopeX - s * slopeY;
            pl.slopeY = s * slopeX + c * slopeY;
            pl.normd = this.normd;
            return pl;
        }

        public Line ClipToBounds(Rect boundingBox)
        {
            ParametricLine xl = new ParametricLine(0, 1, boundingBox.Left, 0);
            ParametricLine yt = new ParametricLine(1, 0, 0, boundingBox.Top);
            ParametricLine xr = new ParametricLine(0, 1, boundingBox.Right, 0);
            ParametricLine yb = new ParametricLine(1, 0, 0, boundingBox.Bottom);
            Point xlp = this.Intersect(xl);
            Point ytp = this.Intersect(yt);
            Point xrp = this.Intersect(xr);
            Point ybp = this.Intersect(yb);
            Point? pt1 = null, pt2 = null;
            if((!VanishingPoint.IsFarAway(xlp)) && (xlp.Y >= boundingBox.Top) && (xlp.Y < boundingBox.Bottom))
            {
                if (pt1 == null)
                {
                    pt1 = xlp;
                }
                else
                {
                    pt2 = xlp;
                }
            }
            if((!VanishingPoint.IsFarAway(ytp)) && (ytp.X >= boundingBox.Left) && (ytp.X < boundingBox.Right))
            {
                if (pt1 == null)
                {
                    pt1 = ytp;
                }
                else
                {
                    pt2 = ytp;
                }
            }
            if((!VanishingPoint.IsFarAway(xrp)) && (xrp.Y >= boundingBox.Top) && (xrp.Y < boundingBox.Bottom))
            {
                if (pt1 == null)
                {
                    pt1 = xrp;
                }
                else
                {
                    pt2 = xrp;
                }
            }
            if((!VanishingPoint.IsFarAway(ybp)) && (ybp.X >= boundingBox.Left) && (ybp.X < boundingBox.Right))
            {
                if (pt1 == null)
                {
                    pt1 = ybp;
                }
                else
                {
                    pt2 = ybp;
                }
            }
            if (pt1 == null || pt2 == null)
            {
                return null;
            }
            Line a = new Line
            {
                X1 = pt1.Value.X,
                Y1 = pt1.Value.Y,
                X2 = pt2.Value.X,
                Y2 = pt2.Value.Y
            };
            return a;
        }

        public static double GetDistance(Point pt1, Point pt2)
        {
            double dx = pt1.X - pt2.X;
            double dy = pt1.Y - pt2.Y;
            return Math.Sqrt(dx * dx + dy * dy);
        }
    }

    class VanishingPoint
    {
        readonly bool isPoint;
        public bool IsPoint { get { return isPoint; } }
        readonly Point valPt;
        readonly ParametricLine valLine;

        public VanishingPoint(ParametricLine pl1, ParametricLine pl2)
        {
            Point p = pl1.Intersect(pl2);
            if (IsFarAway(p))
            {
                isPoint = false;
                valLine = pl1.AverageSlope(pl2, Math.Abs(pl1.AngleTo(pl2)) < 0.1);
            }
            else
            {
                isPoint = true;
                valPt = p;
            }
        }

        public Point GetPoint() {
            return valPt;
        }

        public static ParametricLine GetHorizonLine(VanishingPoint vp1, VanishingPoint vp2)
        {
            if (vp1.IsPoint)
            {
                if (vp2.IsPoint)
                {
                    return new ParametricLine(vp1.valPt, vp2.valPt);
                }
                else
                {
                    return new ParametricLine(vp2.valLine, vp1.valPt);
                }
            }
            else
            {
                if (vp2.IsPoint)
                {
                    return new ParametricLine(vp1.valLine, vp2.valPt);
                }
                else
                {
                    return vp1.valLine.AverageSlope(vp2.valLine, false);
                }
            }
        }

        public ParametricLine ProjectThrough(Point pt)
        {
            if (isPoint)
            {
                return new ParametricLine(pt, valPt);
            }
            else
            {
                return new ParametricLine(valLine, pt);
            }
        }

        public Point ProjectTo(ParametricLine imgLn, Point obj)
        {
            if (isPoint)
            {
                return imgLn.ProjectOntoFromPoint(valPt, obj);
            }
            else
            {
                return imgLn.ProjectOntoFromSlope(valLine, obj);
            }
        }

        public static bool IsFarAway(Point p)
        {
            if (double.IsNaN(p.X) || double.IsInfinity(p.X) || double.IsNaN(p.Y) || double.IsInfinity(p.Y))
            {
                return true;
            }
            double dr = 4000; //maximum expected size of image
            dr *= dr;
            return (Math.Abs(p.X) > dr || Math.Abs(p.Y) > dr);
        }
    }
}
