﻿using Microsoft.Win32;
using System;
using System.Windows;
using System.Windows.Controls;
using System.Windows.Media;
using System.Windows.Media.Imaging;
using System.Windows.Shapes;

//Resources used to make the algorithm:
// https://www.math.utah.edu/~treiberg/Perspect/Perspect.htm
// https://www.handprint.com/HP/WCL/perspect3.html
// https://math.stackexchange.com/questions/3980660/properties-of-measuring-points-in-perspective-drawing

namespace Photogrammetry
{
    /// <summary>
    /// Interaction logic for MainWindow.xaml
    /// </summary>
    public partial class MainWindow : Window
    {
        string saveDir;
        double mag_ratio = 16;
        Point selectedPixel = new Point();
        bool mouseDragValid = false;
        bool spaceKeySwap = false;

        public MainWindow()
        {
            InitializeComponent();
            selectedPixel.X = 0;
            selectedPixel.Y = 0;

            ToolTabs.Items.Add(new TabItem
            {
                Header = "Mode 0",
                Content = new LineProjControl(MainCanvas)
            });

            ToolTabs.Items.Add(new TabItem
            {
                Header = "Mode 1",
                Content = new HorizonProjControl(MainCanvas)
            });

            ToolTabs.Items.Add(new TabItem
            {
                Header = "Mode 2",
                Content = new Plane1ProjControl(MainCanvas)
            });

            ToolTabs.Items.Add(new TabItem
            {
                Header = "Mode 2a",
                Content = new Plane2ProjControl(MainCanvas)
            });


            /*ToolTabs.Items.Add(new TabItem
            {
                Header = "Mode 3a",
                Content = new Cube2ProjControl(MainCanvas)
            });*/

            string[] args = Environment.GetCommandLineArgs();
            if (args.Length == 2)
            {
                loadImage(args[1]);
            }
        }

        private void resetUI()
        {
            //throw new NotImplementedException();
            ImagePathText.Text = "Image Not Loaded";
            //MainCanvas.Children.Clear();
            MainCanvas.Background = null;
            //ToolTabs.Items.Clear();
            MagnifiedCanvas.Children.Clear();
        }

        private void OpenImage_Click(object sender, RoutedEventArgs e)
        {
            OpenFileDialog dlg = new OpenFileDialog();
            if (saveDir != null)
                dlg.InitialDirectory = saveDir;
            else
                dlg.InitialDirectory = Environment.GetFolderPath(Environment.SpecialFolder.MyPictures);
            dlg.Filter = "Image files|*.jpg;*.png;*.bmp|All Files|*.*";
            dlg.RestoreDirectory = true;

            if (dlg.ShowDialog() != true)
                return;
            
            resetUI();

            string selectedFileName = dlg.FileName;
            loadImage(selectedFileName);
        }

        private void loadImage(string filename)
        {
            ImagePathText.Text = filename;
            saveDir = System.IO.Path.GetDirectoryName(filename);
            BitmapImage originalImage = new BitmapImage();
            originalImage.BeginInit();
            originalImage.UriSource = new Uri(filename);
            originalImage.EndInit();
            ImageBrush ib = new ImageBrush();
            ib.ImageSource = originalImage;
            ib.Stretch = Stretch.UniformToFill;
            MainCanvas.Width = originalImage.PixelWidth;
            MainCanvas.Height = originalImage.PixelHeight;
            MainCanvas.Background = ib;

            Image MagnifiedImage = new Image();
            RenderOptions.SetBitmapScalingMode(MagnifiedImage, BitmapScalingMode.NearestNeighbor);
            MagnifiedImage.Source = originalImage;
            MagnifiedImage.Width = originalImage.PixelWidth * mag_ratio;
            MagnifiedImage.Height = originalImage.PixelHeight * mag_ratio;
            MagnifiedCanvas.Children.Add(MagnifiedImage);

            addMagnifierLines();
        }

        private void PasteImage_Click(object sender, RoutedEventArgs e)
        {
            if (!System.Windows.Clipboard.ContainsImage())
            {
                return;
            }
            BitmapSource src = System.Windows.Clipboard.GetImage();

            resetUI();

            string selectedFileName = "Clipboard";
            ImagePathText.Text = selectedFileName;
            //saveDir = System.IO.Path.GetDirectoryName(selectedFileName);
            /*originalImage = new BitmapImage();
            originalImage.BeginInit();
            originalImage.StreamSource =;
            originalImage.EndInit();*/
            ImageBrush ib = new ImageBrush();
            ib.ImageSource = src;
            ib.Stretch = Stretch.UniformToFill;
            MainCanvas.Width = src.PixelWidth;
            MainCanvas.Height = src.PixelHeight;
            MainCanvas.Background = ib;

            Image MagnifiedImage = new Image();
            RenderOptions.SetBitmapScalingMode(MagnifiedImage, BitmapScalingMode.NearestNeighbor);
            MagnifiedImage.Source = src;
            MagnifiedImage.Width = src.PixelWidth * mag_ratio;
            MagnifiedImage.Height = src.PixelHeight * mag_ratio;
            MagnifiedCanvas.Children.Add(MagnifiedImage);

            addMagnifierLines();
        }

        private void addMagnifierLines()
        {
            Line magh = new Line();
            magh.X1 = 0;
            magh.Y1 = MagnifiedCanvas.ActualHeight / 2;
            magh.X2 = MagnifiedCanvas.ActualWidth;
            magh.Y2 = MagnifiedCanvas.ActualHeight / 2;
            magh.Stroke = Brushes.Red;
            MagnifiedCanvas.Children.Add(magh);
            Line magv = new Line();
            magv.Y1 = 0;
            magv.X1 = MagnifiedCanvas.ActualWidth / 2;
            magv.Y2 = MagnifiedCanvas.ActualHeight;
            magv.X2 = MagnifiedCanvas.ActualWidth / 2;
            magv.Stroke = Brushes.Red;
            MagnifiedCanvas.Children.Add(magv);
            selectedPixel = new Point(0, 0);
            Label pixn = new Label();
            pixn.Content = "(0, 0)";
            MagnifiedCanvas.Children.Add(pixn);
            Canvas.SetLeft(pixn, 5);
            Canvas.SetTop(pixn, 5);
            redrawMagnifier();

            /*
            //TransformGroup transformGroup = new TransformGroup();
            ScaleTransform scale = new ScaleTransform();
            scale.CenterX = 0;
            scale.CenterY = 0;
            scale.ScaleX = 2;
            scale.ScaleY = 2;
            MagnifiedImage.Source = originalImage;
            MagnifiedImage.RenderTransform = scale;
            */
        }

        private void MainCanvas_MouseDown(object sender, System.Windows.Input.MouseButtonEventArgs e)
        {
            //this gets called if no child elements (lines or points) handled the event
            /*if (MainCanvas == null)
                return; //no image loaded*/

            spaceKeySwap = (e.RightButton == System.Windows.Input.MouseButtonState.Pressed);

            Point clp = e.GetPosition(MainCanvas);
            selectedPixel.X = Math.Floor(clp.X);
            selectedPixel.Y = Math.Floor(clp.Y);

            updateSelectionPoint();

            if (EnableDrag.IsChecked == true)
            {
                MainCanvas.CaptureMouse();
                mouseDragValid = true;
            }
        }

        void updateSelectionPoint()
        {
            if (ToolTabs.SelectedIndex > 0)
            {
                ((ProjControlBase)((TabItem)ToolTabs.SelectedItem).Content).ApplyTool(selectedPixel, spaceKeySwap);
            }

            redrawMagnifier();
        }

        private void redrawMagnifier()
        {
            if (double.IsNaN(MainCanvas.Width))
                return; //no image loaded

            Canvas.SetLeft((Image)(MagnifiedCanvas.Children[0]), -selectedPixel.X * mag_ratio + MagnifiedCanvas.ActualWidth / 2 - mag_ratio / 2);
            Canvas.SetTop((Image)(MagnifiedCanvas.Children[0]), -selectedPixel.Y * mag_ratio + MagnifiedCanvas.ActualHeight / 2 - mag_ratio / 2);
            
            Label pixn = (Label)(MagnifiedCanvas.Children[3]);
            pixn.Content = selectedPixel.ToString();
        }

        private void Overall_PreviewKeyDown(object sender, System.Windows.Input.KeyEventArgs e)
        {
            if (double.IsNaN(MainCanvas.Width))
                return; //no image loaded

            //handle arrow keys - adjust by pixels
            switch (e.Key)
            {
                case System.Windows.Input.Key.Space:
                    spaceKeySwap = !spaceKeySwap;
                    if (ToolTabs.SelectedIndex > 0)
                    {
                        selectedPixel = ((ProjControlBase)((TabItem)ToolTabs.SelectedItem).Content).GetSelectionPt(spaceKeySwap);
                        redrawMagnifier();
                    }
                    e.Handled = true;
                    return;
                    break;
                case System.Windows.Input.Key.Left:
                    if(selectedPixel.X > 0.5)
                        selectedPixel.X -= 1;
                    break;
                case System.Windows.Input.Key.Right:
                    if (selectedPixel.X < (MainCanvas.Width - 1.5))
                        selectedPixel.X += 1;
                    break;
                case System.Windows.Input.Key.Up:
                    if (selectedPixel.Y > 0.5)
                        selectedPixel.Y -= 1;
                    break;
                case System.Windows.Input.Key.Down:
                    if (selectedPixel.Y < (MainCanvas.Height - 1.5))
                        selectedPixel.Y += 1;
                    break;
                default:
                    e.Handled = false;
                    return;
            }
            updateSelectionPoint();
            e.Handled = true;
        }

        private void ToolTabs_SelectionChanged(object sender, SelectionChangedEventArgs e)
        {
            /*if (ToolTabs.SelectedIndex > 0)
            {
                selectedPixel = ((ProjControlBase)((TabItem)ToolTabs.SelectedItem).Content).GetSelectionPt(spaceKeySwap);
            }
            redrawMagnifier();*/
            e.Handled = true;
        }

        private void MagRadio_Checked(object sender, RoutedEventArgs e)
        {
            RadioButton sdr = (RadioButton)sender;
            if (sdr.Content == null)
                return;
            mag_ratio = double.Parse((string)sdr.Content);
            if (double.IsNaN(MainCanvas.Width))
                return; //no image loaded
            Image mi = (Image)(MagnifiedCanvas.Children[0]);
            mi.Width = MainCanvas.Width * mag_ratio;
            mi.Height = MainCanvas.Height * mag_ratio;
            redrawMagnifier();
        }

        private void MainCanvas_MouseMove(object sender, System.Windows.Input.MouseEventArgs e)
        {
            if (mouseDragValid)
            {
                Point clp = e.GetPosition(MainCanvas);
                selectedPixel.X = Math.Floor(clp.X);
                selectedPixel.Y = Math.Floor(clp.Y);
                updateSelectionPoint();
            }
            e.Handled = true;
        }

        private void MainCanvas_MouseUp(object sender, System.Windows.Input.MouseButtonEventArgs e)
        {
            if (mouseDragValid)
            {
                mouseDragValid = false;
                MainCanvas.ReleaseMouseCapture();
            }
            if (spaceKeySwap)
            {
                OverallScroll.Focus(); //focus on right click too
            }
            e.Handled = true;
        }

        private void MainCanvas_MouseLeave(object sender, System.Windows.Input.MouseEventArgs e)
        {
            if (mouseDragValid)
            {
                mouseDragValid = false;
                MainCanvas.ReleaseMouseCapture();
            }
            e.Handled = true;
        }

        private void MagnifiedCanvas_SizeChanged(object sender, SizeChangedEventArgs e)
        {
            if(MagnifiedCanvas.Children == null || MagnifiedCanvas.Children.Count < 3)
            {
                return;
            }
            Line magh = (Line)(MagnifiedCanvas.Children[1]);
            magh.X1 = 0;
            magh.Y1 = MagnifiedCanvas.ActualHeight / 2;
            magh.X2 = MagnifiedCanvas.ActualWidth;
            magh.Y2 = MagnifiedCanvas.ActualHeight / 2;
            Line magv = (Line)(MagnifiedCanvas.Children[2]);
            magv.Y1 = 0;
            magv.X1 = MagnifiedCanvas.ActualWidth / 2;
            magv.Y2 = MagnifiedCanvas.ActualHeight;
            magv.X2 = MagnifiedCanvas.ActualWidth / 2;
            redrawMagnifier();
        }
    }
}
