﻿using System;
using System.Collections.Generic;
using System.Windows;
using System.Windows.Controls;
using System.Windows.Media;
using System.Windows.Shapes;

namespace Photogrammetry
{
    /// <summary>
    /// Interaction logic for HorizonProjControl.xaml
    /// </summary>
    public partial class HorizonProjControl: ProjControlBase
    {
        ProjectedPlane M2p = null;
        string activeTool = null;
        
        public HorizonProjControl(Canvas mainCanvas) : base(mainCanvas)
        {
            InitializeComponent();
        }

        private void ToolButton_Checked(object sender, RoutedEventArgs e)
        {
            RadioButton snd = (RadioButton)sender;
            SetTool((string)snd.Content);
            e.Handled = true;
        }

        private void Scale_TextChanged(object sender, TextChangedEventArgs e)
        {
            runComputation();
        }

        public override void SetTool(string toolName)
        {
            if (double.IsNaN(myCanvas.Width))
            {
                return;
            }
            activeTool = toolName;
            if (M2p == null)
            {
                M2p = new ProjectedPlane();
                M2p.axis1 = new ProjectedAxis();
                M2p.axis2 = new ProjectedAxis();
            }
            switch (toolName)
            {
                case "Horizon":
                    if (M2p.axis2.primary == null)
                    {
                        M2p.axis2.primary = MakeLine(ZAxisBrush, true);
                    }
                    break;
                case "Ref":
                    if (M2p.axis2.secondary == null)
                    {
                        M2p.axis2.secondary = MakeLine(RefLineBrush);
                    }
                    break;
                case "Meas":
                    if (M2p.measLine == null)
                    {
                        M2p.measLine = MakeLine(MeasLineBrush);
                    }
                    break;
                default:
                    activeTool = null;
                    break;
            }
        }

        public override Point GetSelectionPt(bool rightClick)
        {
            if (activeTool == null || myCanvas == null)
            {
                return new Point();
            }
            if (M2p != null)
            {
                switch (activeTool)
                {
                    case "Horizon":
                        if (M2p.axis2.primary != null)
                        {
                            return GetLineEnd(M2p.axis2.primary, !rightClick);
                        }
                        break;
                    case "Ref":
                        if (M2p.axis2.secondary != null)
                        {
                            return GetLineEnd(M2p.axis2.secondary, !rightClick);
                        }
                        break;
                    case "Meas":
                        if (M2p.measLine != null)
                        {
                            return GetLineEnd(M2p.measLine, !rightClick);
                        }
                        break;
                }
            }
            return new Point();
        }

        public override void ApplyTool(Point selectPt, bool rightClick)
        {
            if (M2p != null)
            {
                switch (activeTool)
                {
                    case "Horizon":
                        if (M2p.axis2.primary != null)
                        {
                            AdjustLine(M2p.axis2.primary, !rightClick, selectPt);
                        }
                        break;
                    case "Ref":
                        if (M2p.axis2.secondary != null)
                        {
                            AdjustLine(M2p.axis2.secondary, !rightClick, selectPt);
                        }
                        break;
                    case "Meas":
                        if (M2p.measLine != null)
                        {
                            AdjustLine(M2p.measLine, !rightClick, selectPt);
                        }
                        break;
                }
                runComputation();
            }
        }

        private void UserControl_IsVisibleChanged(object sender, DependencyPropertyChangedEventArgs e)
        {
            Show((bool)e.NewValue);
            if((bool)e.NewValue == true)
            {
                //reset tool and update event handler
            }
        }

        private void runComputation()
        {
            if (!IsInitialized)
            {
                return;
            }

            if (M2p == null || M2p.axis2.primary == null || M2p.axis2.secondary == null || M2p.measLine == null)
            {
                M1aresl.Text = "L";
                M1aresa.Text = "A";
                M1aresp.Text = "P";
                return;
            }

            double wafov;
            if (!double.TryParse(M1afov.Text, out wafov))
            {
                wafov = double.NaN;
            }
            double reflen;
            if (!double.TryParse(M1arefl.Text, out reflen))
            {
                reflen = double.NaN;
            }

            ParametricLine horizon = new ParametricLine(M2p.axis2.primary);
            Point img_origin = new Point(myCanvas.Width / 2, myCanvas.Height / 2);
            ParametricLine virtualx = new ParametricLine(horizon.Perpendicular(), img_origin);
            Point intercept = virtualx.Intersect(horizon);
            double vx = ParametricLine.GetDistance(img_origin, intercept);
            double fov = wafov / 180 * Math.PI;
            double di = (Math.Max(myCanvas.Width, myCanvas.Height) / 2) / Math.Tan(fov / 2);
            double phi = Math.Atan2(di, vx);
            double sp = di / Math.Sin(phi);
            ParametricLine measuring = new ParametricLine(horizon, intercept);
            Point dvp1 = measuring.Traverse(sp);
            Point dvp2 = measuring.Traverse(-sp);
            Point meas_offset = new ParametricLine(virtualx, intercept).Traverse(Math.Max(myCanvas.Width, myCanvas.Height) / 2);
            ParametricLine meas_line = new ParametricLine(horizon, meas_offset);
            //project the reference points onto corresponding axes
            Point rx1, rx2, ry1, ry2;
            Point ref1 = GetLineEnd(M2p.axis2.secondary, true);
            Point ref2 = GetLineEnd(M2p.axis2.secondary, false);
            rx1 = meas_line.ProjectOntoFromPoint(dvp1, ref1);
            ry1 = meas_line.ProjectOntoFromPoint(dvp2, ref1);
            rx2 = meas_line.ProjectOntoFromPoint(dvp1, ref2);
            ry2 = meas_line.ProjectOntoFromPoint(dvp2, ref2);
            ParametricLine rpx = new ParametricLine(rx1, rx2);
            ParametricLine rpy = new ParametricLine(ry1, ry2);
            double rdx = rpx.DotProduct(meas_line) > 0 ? rpx.NormDist : (-rpx.NormDist);
            double rdy = rpy.DotProduct(meas_line) > 0 ? rpy.NormDist : (-rpy.NormDist);
            ParametricLine rpc = new ParametricLine(rdx, rdy, 0, 0);
            Point mx1, mx2, my1, my2;
            Point mmp1 = GetLineEnd(M2p.measLine, true);
            Point mmp2 = GetLineEnd(M2p.measLine, false);
            mx1 = meas_line.ProjectOntoFromPoint(dvp1, mmp1);
            my1 = meas_line.ProjectOntoFromPoint(dvp2, mmp1);
            mx2 = meas_line.ProjectOntoFromPoint(dvp1, mmp2);
            my2 = meas_line.ProjectOntoFromPoint(dvp2, mmp2);
            ParametricLine mpx = new ParametricLine(mx1, mx2);
            ParametricLine mpy = new ParametricLine(my1, my2);
            double mdx = mpx.DotProduct(meas_line) > 0 ? mpx.NormDist : (-mpx.NormDist);
            double mdy = mpy.DotProduct(meas_line) > 0 ? mpy.NormDist : (-mpy.NormDist);
            ParametricLine ref_obj = new ParametricLine(rdx, rdy, 0, 0);
            ParametricLine query_obj = new ParametricLine(mdx, mdy, 0, 0);
            double rlen = (query_obj.NormDist / ref_obj.NormDist) * reflen;
            double rang = -ref_obj.AngleTo(query_obj) / Math.PI * 180; //[deg]
            double pang = phi / Math.PI * 180;
            M1aresl.Text = "L : " + rlen.ToString("G4") + " <len>";
            M1aresa.Text = "A : " + rang.ToString("G4") + " [deg]";
            M1aresp.Text = "P : " + pang.ToString("G4") + " [deg]";
        }

        private void ClearVP_Button_Click(object sender, RoutedEventArgs e)
        {

        }
    }
}
