﻿using System.Text;

namespace MCalcConsole
{
    internal class Units
    {
        SortedList<string, (double, string)>? myUnits = null;
        string ul_filename;

        public Units(string filename)
        {
            ul_filename = filename;
        }

        void ParseUnitsList()
        {
            string[] lines = File.ReadAllLines(ul_filename);
            myUnits = new SortedList<string, (double, string)>();
            foreach (string line in lines)
            {
                if (line.Length == 0) continue;
                if (!char.IsLetter(line[0])) continue;
                string[] lr = line.Split('=');
                if (lr.Length != 2) throw new ArgumentException("Incorrect use of equal sign: " + line);
                string un = lr[0].Trim();
                if (string.IsNullOrEmpty(un)) continue;
                if (!IsAllLetters(un)) throw new ArgumentException("Incorrect unit name 1: " + line);
                string ud = lr[1].Trim();
                string[] nu = ud.Split(' ');
                double num = double.Parse(nu[0]);
                if (nu.Length < 2)
                {
                    ud = "";
                }
                else
                {
                    ud = nu[1].Trim();
                }
                if (un == ud && num != 1) throw new ArgumentException("Self-reference unit must have value 1: " + line);
                if (!double.IsFinite(num) || num == 0) throw new ArgumentException("Incorrect value for unit definition: " + line);
                myUnits.Add(un, (num, ud));
            }
        }

        static bool IsAllLetters(string str)
        {
            if (str == null) return false;
            for (int i = 0; i < str.Length; i++)
            {
                if (!char.IsLetter(str[i])) return false;
            }
            return true;
        }

        public Unit ReduceToBase(Unit uin)
        {
            if (myUnits == null)
                ParseUnitsList();
            List<string> ls = new List<string>();
            return ReduceToBase(uin, ref ls);
        }

        Unit ReduceToBase(Unit uin, ref List<string> usl)
        {
            if (myUnits == null)
                throw new Exception("Units list was not loaded");
            Unit uc = new Unit();
            uc.Multiplier = uin.Multiplier;
            //SortedSet<string> usl = new SortedSet<string>();
            int uslst = usl.Count;
            foreach (KeyValuePair<string, double> kvp in uin)
            {
                (double, string) mu;
                if (!myUnits.TryGetValue(kvp.Key, out mu)) throw new Exception("Unknown unit: " + kvp.Key);
                if (kvp.Key == mu.Item2) //base unit
                {
                    if (mu.Item1 != 1) throw new Exception("Incorrect base unit 1 definition: " + kvp.Key);
                    uc.MultiplyBy(kvp.Key, kvp.Value);
                    continue;
                }
                if (usl.Contains(kvp.Key)) throw new Exception("Units circular reference: " + kvp.Key);
                usl.Add(kvp.Key);
                Unit nbu = new Unit(mu.Item1, mu.Item2);
                nbu.Exponentiate(kvp.Value);
                uc.MultiplyBy(ReduceToBase(nbu, ref usl));
                usl.RemoveAt(uslst);
            }
            return uc;
        }

        public Unit ConvertToUnit(Unit uin, string uout)
        {
            if (myUnits == null)
                ParseUnitsList();
            uin = ReduceToBase(uin);
            Unit un = new Unit(1, uout);
            Unit uo = ReduceToBase(un);
            if (!uin.Matches(uo)) throw new Exception("Units are not compatible");
            un.Multiplier = uin.Multiplier / uo.Multiplier;
            return un;
        }
    }

    internal class Unit : SortedList<string, double>
    {
        public double Multiplier = 1;
        
        public Unit() { }

        string? prefdisp;
        
        public Unit(double mult, string ustr)
        {
            Multiplier = mult;
            int l = ustr.Length;
            bool umult = true, upwr = false;
            StringBuilder sb = new StringBuilder();
            StringBuilder sc = new StringBuilder();
            double sv;
            string ss = "";
            for (int i = 0; i <= l; i++)
            {
                char c = (i < l) ? ustr[i] : '*';
                if (c == '/' || c == '*')
                {
                    if (upwr)
                    {
                        sv = double.Parse(sb.ToString());
                    }
                    else
                    {
                        sv = 1;
                        ss = sb.ToString();
                    }
                    if (!string.IsNullOrEmpty(ss) && sv != 0)
                    {
                        this.MultiplyBy(ss, (umult ? sv : -sv));
                        if (sc.Length > 0 || !umult)
                            sc.Append(umult ? '*' : '/');
                        sc.Append(ss);
                        if (sv != 1)
                        {
                            sc.Append('^');
                            sc.Append(sv);
                        }
                    }
                    sb.Clear();
                    umult = (c == '*');
                    upwr = false;
                    continue;
                }
                if (c == '^')
                {
                    upwr = true;
                    ss = sb.ToString();
                    sb.Clear();
                    continue;
                }
                if (!upwr && !char.IsLetter(c))
                    throw new Exception("Incorrect symbol in unit name: " + ustr);
                sb.Append(c);
            }
            prefdisp = sc.ToString();
        }

        /*public Unit Copy()
        {
            Unit r = new Unit();
            foreach(KeyValuePair<string, double> kvp in this)
            {
                r.Add(kvp.Key, kvp.Value);
            }
            r.Multiplier = this.Multiplier;
            return r;
        }*/

        public bool Matches(Unit ut)
        {
            int j = this.Count;
            if(j != ut.Count) return false;
            for(int i=0; i<j; i++)
            {
                if(this.GetKeyAtIndex(i) != ut.GetKeyAtIndex(i)) return false;
                if (this.GetValueAtIndex(i) != ut.GetValueAtIndex(i)) return false;
            }
            return true;
        }

        public void MultiplyBy(string un, double pow)
        {
            double ev;
            if (string.IsNullOrEmpty(un)) throw new Exception("Empty unit name may not be added to unit stack");
            if (this.TryGetValue(un, out ev))
            {
                ev += pow;
                if (ev != 0 && double.IsFinite(ev))
                    this[un] = ev;
                else
                    this.Remove(un);
            }
            else
            {
                if(pow != 0 && double.IsFinite(pow))
                    this.Add(un, pow);
            }
        }

        public void Exponentiate(double pow)
        {
            if (!double.IsFinite(pow) || pow == 0) throw new Exception("Unsupported power value: " + pow.ToString());
            if (pow == 1) return;
            Multiplier = Math.Pow(Multiplier, pow);
            int j = this.Count;
            for(int i=0; i<j; i++)
            {
                this.SetValueAtIndex(i, pow * this.GetValueAtIndex(i));
            }
        }

        public void MultiplyBy(Unit mult)
        {
            foreach (KeyValuePair<string, double> kvp in mult)
            {
                this.MultiplyBy(kvp.Key, kvp.Value);
            }

            this.Multiplier *= mult.Multiplier;
        }

        public override string ToString()
        {
            StringBuilder sb = new StringBuilder();
            int i = 0;
            //if(Multiplier != 1)
            //{
                sb.Append(Multiplier);
                sb.Append(' ');
            //}

            if (prefdisp != null)
            {
                sb.Append(prefdisp);
            }
            else
            {
                foreach (KeyValuePair<string, double> kvp in this)
                {
                    if (i > 0 || kvp.Value <= 0)
                    {
                        sb.Append((kvp.Value > 0) ? '*' : '/');
                    }
                    sb.Append(kvp.Key);
                    double v = Math.Abs(kvp.Value);
                    if (v != 1)
                    {
                        sb.Append('^');
                        sb.Append(kvp.Value);
                    }
                    i++;
                }
            }
            return sb.ToString();
        }
    }
}
