﻿using System.Collections.Immutable;
using System.IO;
using System.Runtime.Intrinsics.X86;

namespace MCalcConsole
{
    internal class Processor
    {
        internal required StackEntriesConsole myStack;
        internal ExtCmdManager myExtCmds = new ExtCmdManager();
        internal Units myUnits = new Units(units_file);
        SortedSet<string> runningExtCmds = new SortedSet<string>();
        const string ml_type = " :", it_type = " {}";
        const char vc_sep = '|';
        public const string ve_cmd = "@", vq_cmd = "?", exit_cmd = "esc", vc_cmd = "conv", ei_cmd = "in", eo_cmd = "out", units_file = "units_list.txt";

        public Processor()
        {
            
        }
        
        public bool ProcessMetaCommand(InputCommand cmdIn, out string? entryText, int atBuffer)
        {
            //check meta commands
            entryText = null;
            int rl;
            switch (cmdIn.cmdword)
            {
                case "b":
                    Program.ClearLine(atBuffer);
                    if (cmdIn.cmdrefs == null || cmdIn.cmdrefs.Count < 1)
                    {
                        rl = myStack.TrimToLength(myStack.Count - 1, atBuffer - 1);
                    }
                    else
                    {
                        if (cmdIn.cmdrefs.Count != 1 || cmdIn.cmdrefs[0] is not SRSingle)
                            throw new Exception("b can use only one reference");
                        SRSingle srs = (SRSingle)cmdIn.cmdrefs[0];
                        rl = myStack.TrimToLength(srs.stackIndex, atBuffer - 1);
                    }
                    rl++;
                    Console.SetCursorPosition(Program.layout_indices[1], rl > 0 ? rl:0);
                    return true;
                case "cs":
                    if (cmdIn.cmdrefs != null && cmdIn.cmdrefs.Count > 0)
                        throw new Exception("cs cannot contain references");
                    rl = myStack.TrimToLength(0, atBuffer - 1);
                    Console.SetCursorPosition(Program.layout_indices[1], 0);
                    return true;
                case "reload":
                    if (cmdIn.cmdrefs != null && cmdIn.cmdrefs.Count > 0)
                        throw new Exception("reload cannot contain references");
                    myExtCmds = new ExtCmdManager();
                    runningExtCmds.Clear();
                    myUnits = new Units(units_file);
                    Console.CursorLeft = Program.layout_indices[1];
                    return true;
                case "listcmds":
                    if (string.IsNullOrEmpty(cmdIn.Comment))
                        throw new Exception("No file name provided");
                    using (FileStream stream = File.OpenWrite(cmdIn.Comment))
                    {
                        using (StreamWriter sw = new StreamWriter(stream))
                        {
                            foreach (KeyValuePair<string, CmdProps> kv in Maths.mathCmds)
                            {
                                sw.Write(kv.Key);
                                sw.Write('\t');
                                sw.Write(kv.Value.minargs);
                                if (kv.Value.maxargs != kv.Value.minargs)
                                {
                                    if (kv.Value.maxargs == int.MaxValue)
                                    {
                                        sw.Write('+');
                                    }
                                    else
                                    {
                                        sw.Write('-');
                                        sw.Write(kv.Value.maxargs);
                                    }
                                }
                                sw.Write('\t');
                                sw.Write(kv.Value.outlines);
                                sw.Write('\t');
                                sw.Write(kv.Value.description);
                                sw.WriteLine();
                            }
                        }
                    }
                    Console.CursorLeft = Program.layout_indices[1];
                    return true;
                case "export":
                    if (string.IsNullOrEmpty(cmdIn.Comment))
                        throw new Exception("No file name provided");
                    using(FileStream stream = File.OpenWrite(cmdIn.Comment))
                    {
                        using (StreamWriter sw = new StreamWriter(stream))
                        {
                            int n = 0;
                            double v;
                            foreach (StackEntry se in myStack)
                            {
                                sw.Write(se.TypedEntry);
                                if (myStack.Values.TryGetValue(n, out v))
                                {
                                    sw.Write('\t');
                                    sw.Write(v);
                                }
                                if(se.Comment != null)
                                {
                                    sw.Write('\t');
                                    sw.Write(se.Comment);
                                }
                                sw.WriteLine();
                            }
                        }
                    }
                    Console.CursorLeft = Program.layout_indices[1];
                    return true;
            }
            return false;
        }

        public List<CommandSE> UnpackInput(InputCommand cmdIn, StackEntries sse, int i0)
        {
            if (cmdIn.cmdword == vc_cmd) //special treatment to add units in command
            {
                if (string.IsNullOrEmpty(cmdIn.Comment))
                    throw new ArgumentException("Unit conversion string not specified");
                string[] ustr = cmdIn.Comment.Split(' ');
                if (ustr.Length < 2)
                    throw new ArgumentException("Could not parse unit conversion string");
                cmdIn.sarg = ustr[0] + vc_sep + ustr[1];
                cmdIn.Comment = string.Join(' ', ustr.Skip(1));
            }
            List<CommandSE> unpackedCmds;
            //check iterators
            if (cmdIn.cmdrefs != null && cmdIn.cmdrefs.Count > 0)
            {
                int itl = 1, itr;
                foreach(StackReference sr in cmdIn.cmdrefs)
                {
                    if(sr is SRIterator)
                    {
                        itr = ((SRIterator)sr).refs.Count;
                        if (itr == 0)
                            throw new Exception("Zero length iterator");
                        if(itr == 1)
                        {
                            continue;
                        }
                        else
                        {
                            if (itl != 1 && itr != itl)
                                throw new Exception("Command iterators are not compatible");
                            itl = itr;
                        }
                    }
                }
                unpackedCmds = new List<CommandSE>(itl);
                for(int i=0; i<itl; i++)
                {
                    CommandSE cs = new CommandSE() { cmdword = cmdIn.cmdword, sarg = cmdIn.sarg, mlin = i, prefs = cmdIn.prefs };
                    List<SRSingle> ls = new List<SRSingle>();
                    foreach (StackReference sr in cmdIn.cmdrefs)
                    {
                        SRSingle asr;
                        if (sr is SRIterator)
                        {
                            SRIterator si = ((SRIterator)sr);
                            if (si.refs.Count == 1)
                            {
                                asr = si.refs[0];
                            }
                            else
                            {
                                asr = si.refs[i];
                            }
                        }
                        else
                        {
                            asr = (SRSingle)sr;
                        }
                        if (asr.stackIndex >= i0)
                            throw new Exception("Future references not supported");
                        ls.Add(asr);
                    }
                    cs.cmdrefs = ls;
                    if (i == 0)
                    {
                        cs.TypedEntry = cmdIn.TypedEntry;
                        cs.Comment = cmdIn.Comment;
                    }
                    else
                    {
                        cs.TypedEntry = it_type;
                        //cs.Comment = cmdIn.Comment;
                    }
                    unpackedCmds.Add(cs);
                }
            }
            else
            {
                unpackedCmds = new List<CommandSE>() { new CommandSE() { cmdword = cmdIn.cmdword, sarg = cmdIn.sarg, cmdrefs = null, Comment = cmdIn.Comment, TypedEntry = cmdIn.TypedEntry, prefs = cmdIn.prefs } };
            }

            //check multi-line commands
            int j = unpackedCmds.Count;
            for (int i=0; i<j; i++)
            {
                CommandSE cs = unpackedCmds[i];
                if (cs.cmdword == ve_cmd)
                { //this needs more work
                    if (cs.cmdrefs == null)
                        throw new Exception(cs.cmdword + " (resolved) requires references");
                    int k = cs.prefs - 1;
                    if (k < 1)
                        throw new Exception(cs.cmdword + " (resolved) requires 2+ references");
                    int vs = k * 2 + 1;
                    PrependEndRefs(i0, ref cs.cmdrefs, vs, vs);
                    //SortedSet<int> knownInds = new SortedSet<int>();
                    int min = cs.cmdrefs[k].stackIndex;
                    double v = double.NaN;
                    List<SRSingle> drefs = new List<SRSingle>();
                    SortedList<int, int> sinds = new SortedList<int, int>();
                    SortedSet<int> knownInds = new SortedSet<int>();
                    for (int l = 0; l < k; l++)
                    {
                        int q = cs.cmdrefs[l + k].stackIndex;
                        if (!knownInds.Add(q))
                            throw new Exception("Value multiply defined");
                        sinds.Add(q, cs.cmdrefs[l].stackIndex);
                        drefs.Add(cs.cmdrefs[l]);
                        if (q < min)
                            min = q;
                    }

                    int qq = cs.cmdrefs.Last().stackIndex;
                    List<(int, int)> tree = BuildDependencyTree(sse, qq, knownInds, min);
                    SortedSet<int> supplyInds = TrimAndCheck(ref tree, knownInds);
                    foreach (int id in supplyInds)
                    {
                        if (!knownInds.Add(id))
                            throw new Exception("Value multiply defined 2");
                        sinds.Add(id, id);
                        drefs.Add(new SRSingle() { stackIndex = id });
                    }

                    List<int> path = BuildSolutionPath(tree);
                    if (path.Count < 1)
                        //return newSC[cmdIn.cmdrefs.Last().stackIndex];
                        throw new Exception("Nothing to solve");

                    unpackedCmds[i] = new HDCommandSE() { cmdword = cs.cmdword, sarg = cs.sarg, prefs = cs.prefs, TypedEntry = cs.TypedEntry, Comment = cs.Comment, cmdrefs = cs.cmdrefs, fwdpath = path, supplyInds = sinds, drefs = drefs };
                    continue;
                }
                if(cs.cmdword == vq_cmd)
                { //this needs more work
                    if (cs.cmdrefs == null)
                        throw new Exception(vq_cmd + " (resolved) requires references");
                    int k = cs.prefs - 1;
                    if (k < 1)
                        throw new Exception(vq_cmd + " (resolved) requires 2+ references");
                    int vs = k * 2 + 1;
                    PrependEndRefs(i0, ref cs.cmdrefs, vs, vs);
                    SortedSet<int> knownInds = new SortedSet<int>();
                    SortedList<int, int> sinds = new SortedList<int, int>();
                    int min = cs.cmdrefs[k].stackIndex;
                    int max = min;
                    double v = double.NaN;
                    List<SRSingle> drefs = new List<SRSingle>();
                    int q;
                    for (int l = 0; l < k; l++)
                    {
                        q = cs.cmdrefs[l + k].stackIndex;
                        if (!knownInds.Add(q))
                            throw new Exception("Value multiply defined");
                        sinds.Add(q, cs.cmdrefs[l].stackIndex);
                        drefs.Add(cs.cmdrefs[l]);
                        if (q < min)
                            min = q;
                        if (q > max)
                            max = q;
                    }
                    q = cs.cmdrefs.Last().stackIndex;
                    if (q < min)
                        min = q;
                    if (q >= max)
                        throw new Exception("Causality mismatch");
                    if (!knownInds.Add(q))
                        throw new Exception("Value multiply defined");
                    knownInds.Remove(max);
                    List<(int, int)> tree = BuildDependencyTree(sse, max, knownInds, min);
                    SortedSet<int> supplyInds = TrimAndCheck(ref tree, knownInds, q);

                    foreach (int id in supplyInds)
                    {
                        sinds.Add(id, id);
                        drefs.Add(new SRSingle() { stackIndex = id });
                    }

                    (List<int>, List<(int, int)>) path = BuildInversePath(tree, q);

                    if (path.Item2.Count <= 0)
                        throw new Exception("Nothing to solve");

                    unpackedCmds[i] = new HDCommandSE() { cmdword = cs.cmdword, sarg = cs.sarg, cmdrefs = cs.cmdrefs, prefs = cs.prefs, TypedEntry = cs.TypedEntry, Comment = cs.Comment, fwdpath = path.Item1, revpath = path.Item2, supplyInds = sinds, fn = max, drefs = drefs };
                    continue;
                }
                int ln = GetCmdLinesRefs(ref cs, i0);
                int lo = 0;
                while(ln > 1)
                {
                    ln--;
                    lo++;
                    unpackedCmds.Insert(i + lo, new CommandSE() { cmdword = cs.cmdword, sarg = cs.sarg, cmdrefs = (cs.cmdrefs!=null?new List<SRSingle>(cs.cmdrefs):null), TypedEntry = ml_type, prefs = cs.prefs, mlin = cs.mlin, mlout = lo });
                }
                i += lo;
                j += lo;
            }

            return unpackedCmds;
        }

        internal int GetCmdLinesRefs(ref CommandSE cmdIn, int i0)
        {
            string cmw = cmdIn.cmdword;
            CmdProps? cmdp;
            if (Maths.mathCmds.TryGetValue(cmw, out cmdp))
            {
                PrependEndRefs(i0, ref cmdIn.cmdrefs, cmdp.minargs, cmdp.maxargs);
                return cmdp.outlines;
            }
            if (myExtCmds.AvailableCommands.Contains(cmw))
            {
                if (!myExtCmds.LoadedCommands.ContainsKey(cmw))
                {
                    if (!runningExtCmds.Add(cmw))
                        throw new Exception("External command line-resolving reference loop or stall");
                    myExtCmds.LoadCommand(cmw, this);
                    runningExtCmds.Remove(cmw);
                }
                ExtCmdProps ecp = myExtCmds.LoadedCommands[cmw];
                PrependEndRefs(i0, ref cmdIn.cmdrefs, ecp.inlines, ecp.inlines);
                return ecp.outlines;
            }
            if (cmw == vc_cmd)
            { //this needs more work
                PrependEndRefs(i0, ref cmdIn.cmdrefs, 1, 1);
                return 1;
            }
            if (cmw == ve_cmd || cmw == vq_cmd)
            {
                throw new Exception("Unpack function bypassed (should never see this)");
            }
            if (cmw == eo_cmd)
            {
                PrependEndRefs(i0, ref cmdIn.cmdrefs, 1, int.MaxValue);
                return 1;
            }
            if (cmw == ei_cmd)
            {
                PrependEndRefs(i0, ref cmdIn.cmdrefs, 0, 0);
                return 1;
            }
            throw new Exception("Unknown command");
        }

        public double ProcessCmd(StackEntries sse, VirtualStack ctxt, CommandSE cmdIn)
        {
            List<double> rv;
            CmdProps? cmdp;
            if (Maths.mathCmds.TryGetValue(cmdIn.cmdword, out cmdp))
            {
                rv = GetValues(ctxt, cmdIn.cmdrefs);
                return cmdp.cmdfunc(rv, cmdp.fnum + cmdIn.mlout, -1);
            }
            if (myExtCmds.AvailableCommands.Contains(cmdIn.cmdword))
            {
                ExtCmdProps ecp = myExtCmds.LoadedCommands[cmdIn.cmdword];
                rv = GetValues(ctxt, cmdIn.cmdrefs);
                List<double> result = ProcessExtStack(cmdIn.cmdword, ecp, rv);
                return result[cmdIn.mlout];
            }
            if (cmdIn.cmdword == vc_cmd)
            {
                if (cmdIn.sarg == null)
                    throw new Exception("Unit conversion string not supplied");
                string[] ustr = cmdIn.sarg.Split(vc_sep);
                rv = GetValues(ctxt, cmdIn.cmdrefs);
                if (ustr.Length != 2)
                    throw new ArgumentException("Could not parse unit conversion string");
                Unit ui = new Unit(rv[0], ustr[0]);
                Unit uo = myUnits.ConvertToUnit(ui, ustr[1]);
                return uo.Multiplier;
            }
            if (cmdIn is HDCommandSE)
            {
                HDCommandSE hdcmd = (HDCommandSE)cmdIn;
                double v = double.NaN;
                VirtualStack newSC = new VirtualStack();
                if (hdcmd.supplyInds == null)
                    throw new Exception("No changes to apply");
                foreach (KeyValuePair<int, int> kvp in hdcmd.supplyInds)
                {
                    newSC.Add(kvp.Key, ctxt[kvp.Value]);
                }
                if(hdcmd.fwdpath != null)
                {
                    foreach (int id in hdcmd.fwdpath)
                    {
                        if (sse[id] is not CommandSE)
                        {
                            throw new Exception("Unreferenced solution path (should never see this)");
                        }
                        CommandSE cse = (CommandSE)sse[id];
                        v = ProcessCmd(sse, newSC, cse);
                        newSC.Add(id, v);
                    }
                }
                if (hdcmd.cmdword == ve_cmd)
                {
                    return v;
                }
                if (hdcmd.cmdword == vq_cmd)
                {
                    v = newSC[hdcmd.fn];
                    foreach ((int id, int arg) in hdcmd.revpath)
                    {
                        if (sse[id] is not CommandSE)
                        {
                            throw new Exception("Unreferenced inverse path (should never see this)");
                        }
                        CommandSE cse = (CommandSE)sse[id];
                        v = InverseCmd(sse, newSC, cse, v, arg);
                        //newSC.Add(cse.cmdrefs[arg].stackIndex, v);
                    }
                    return v;
                }
            }
            throw new Exception("Unknown command type");
        }

        public double InverseCmd(StackEntries sse, VirtualStack ctxt, CommandSE cmdIn, double v, int arg)
        {
            CmdProps? cmdp;
            if (arg < 0)
                throw new Exception("Incorrect arg<0 (should never see this)");
            if (Maths.mathCmds.TryGetValue(cmdIn.cmdword, out cmdp))
            {
                if (!cmdp.inv)
                    throw new Exception("Command not invertible");
                if (cmdIn.cmdrefs == null || cmdIn.cmdrefs.Count < 1)
                    throw new Exception("No arguments to invert");
                List<SRSingle> cr = new List<SRSingle>(cmdIn.cmdrefs);
                cr.RemoveAt(arg);
                List<double> rv = new List<double>() { v };
                rv.AddRange(GetValues(ctxt, cr));
                return cmdp.cmdfunc(rv, cmdp.fnum + cmdIn.mlout, arg);
            }
            if (cmdIn.cmdword == vc_cmd)
            {
                if (cmdIn.sarg == null || arg > 1)
                    throw new Exception("Unit conversion inversion error (should never see this)");
                string[] ustr = cmdIn.sarg.Split(vc_sep);
                if (ustr.Length != 2)
                    throw new ArgumentException("Could not parse unit conversion inversion string");
                Unit ui = new Unit(v, ustr[1-arg]);
                Unit uo = myUnits.ConvertToUnit(ui, ustr[arg]);
                return uo.Multiplier;
            }
            if (myExtCmds.AvailableCommands.Contains(cmdIn.cmdword))
            {
                if (cmdIn.cmdrefs == null || cmdIn.cmdrefs.Count < 1)
                    throw new Exception("No arguments to invert");
                ExtCmdProps ecp = myExtCmds.LoadedCommands[cmdIn.cmdword];
                List<SRSingle> cr = new List<SRSingle>(cmdIn.cmdrefs);
                cr.RemoveAt(arg);
                List<double> rv = GetValues(ctxt, cr);
                return InverseExtStack(cmdIn.cmdword, ecp, rv, arg, v, cmdIn.mlout);
            }
            throw new Exception("Cannot invert command");
        }

        private static void PrependEndRefs(int i0, ref List<SRSingle>? rs, int minimumLength, int maximumLength)
        {
            if (rs == null)
            {
                if (minimumLength == 0)
                    return;
                rs = new List<SRSingle>();
            }
            int q = rs.Count;
            if (q > maximumLength)
                throw new Exception("Too many arguments for command");
            foreach(SRSingle ss in rs)
            {
                if(ss.stackIndex >= i0)
                    throw new Exception("Future or self reference not allowed");
            }
            if (q >= minimumLength)
                return;
            int j = 1;
            do
            {
                if ((i0 - j) < 0)
                    throw new Exception("Not enough entries in stack");
                rs.Insert(0, new SRSingle() { stackIndex = (i0 - j) });
                j++;
                q++;
            } while (q < minimumLength);
        }

        private List<double> GetValues(VirtualStack sse, List<SRSingle>? refs)
        {
            if (refs == null)
                return new List<double>();
            List<double> ol = new List<double>(refs.Count);
            double v;
            foreach (SRSingle sss in refs)
            {
                if (!sse.TryGetValue(sss.stackIndex, out v))
                    throw new Exception("A referenced entry does not have a numeric value");
                ol.Add(v);
            }
            return ol;
        }

        private List<double> ProcessExtStack(string cmname, ExtCmdProps ecp, List<double> inv)
        {
            if (!runningExtCmds.Add(cmname))
                throw new Exception("Circular reference or stall in external command");
            StackEntries ese = ecp.myStack;
            int k = ese.Count;
            int ivi = 0;
            List<double> ouv = new List<double>();
            ese.Values.Clear();
            for(int i=0; i<k; i++)
            {
                StackEntry se = ese[i];
                if(se is CommandSE)
                {
                    CommandSE cse = (CommandSE)se;
                    if (cse.cmdword == Processor.ei_cmd)
                    {
                        ese.Values[i] = inv[ivi];
                        ivi++;
                        continue;
                    }
                    if (cse.cmdword == Processor.eo_cmd)
                    {
                        ouv.AddRange(GetValues(ese.Values, cse.cmdrefs));
                        continue;
                    }
                    ese.Values[i] = ProcessCmd(ese, ese.Values, (CommandSE)se);
                }
                if(se is ValueSE)
                {
                    ese.Values[i] = ((ValueSE)se).value;
                }
            }

            runningExtCmds.Remove(cmname);
            return ouv;
        }

        private double InverseExtStack(string cmname, ExtCmdProps ecp, List<double> inv, int arg, double v, int vl)
        {
            if (!runningExtCmds.Add(cmname))
                throw new Exception("Circular reference or stall in external command");
            StackEntries ese = ecp.myStack;
            int k = ese.Count;
            int ivi = 0;
            List<SRSingle> ov = new List<SRSingle>();
            List<SRSingle> iv = new List<SRSingle>();
            //ese.Values.Clear();
            for (int i = 0; i < k; i++)
            {
                StackEntry se = ese[i];
                if (se is CommandSE)
                {
                    CommandSE cse = (CommandSE)se;
                    if (cse.cmdword == Processor.ei_cmd)
                    {
                        iv.Add(new SRSingle() { stackIndex = i });
                        continue;
                    }
                    if (cse.cmdword == Processor.eo_cmd)
                    {
                        if (cse.cmdrefs == null)
                            throw new Exception("External output unreferenced (should never see this)");
                        ov.AddRange(cse.cmdrefs);
                        continue;
                    }
                }
            }

            //if (cs.cmdrefs == null)
            //throw new Exception(vq_cmd + " (resolved) requires references");
            //int k = cs.prefs - 1;

            inv.Insert(arg, double.NaN);//placeholder

            if (iv.Count < 1 || ov.Count < 1 || arg < 0 || arg >= iv.Count || vl < 0 || vl >= ov.Count || inv.Count != iv.Count)
                throw new Exception("Invalid argument index for inverting external command");
            //int vs = k * 2 + 1;
            //PrependEndRefs(i0, ref cs.cmdrefs, vs, vs);

            SortedSet<int> knownInds = new SortedSet<int>();
            //SortedList<int, int> sinds = new SortedList<int, int>();
            int min = ov[vl].stackIndex;
            knownInds.Add(min);
            int max = min;
            //double v = double.NaN;
            //List<SRSingle> drefs = new List<SRSingle>();
            VirtualStack newSC = new VirtualStack();
            int q;
            k = iv.Count;
            for (int l = 0; l < k; l++)
            {
                if (l == arg)
                    continue;
                q = iv[l].stackIndex;
                //ese.Values[q] = inv[l];
                newSC.Add(q, inv[l]);
                if (!knownInds.Add(q))
                    throw new Exception("Value multiply defined");
                //sinds.Add(q, cs.cmdrefs[l].stackIndex);
                //drefs.Add(cs.cmdrefs[l]);
                if (q < min)
                    min = q;
                if (q > max)
                    max = q;
            }
            q = iv[arg].stackIndex;
            if (q < min)
                min = q;
            if (q >= max)
                throw new Exception("Causality mismatch");
            if (!knownInds.Add(q))
                throw new Exception("Value multiply defined");
            int qo = ov[vl].stackIndex;
            knownInds.Remove(qo);
            List<(int, int)> tree = BuildDependencyTree(ese, qo, knownInds, min);
            SortedSet<int> supplyInds = TrimAndCheck(ref tree, knownInds, q);

            /*foreach (int id in supplyInds)
            {
                sinds.Add(id, id);
                drefs.Add(new SRSingle() { stackIndex = id });
            }*/

            (List<int>, List<(int, int)>) path = BuildInversePath(tree, q);

            if (path.Item2.Count <= 0)
                throw new Exception("Nothing to solve");

            //unpackedCmds[i] = new HDCommandSE() { cmdword = cs.cmdword, sarg = cs.sarg, cmdrefs = cs.cmdrefs, prefs = cs.prefs, TypedEntry = cs.TypedEntry, Comment = cs.Comment, fwdpath = path.Item1, revpath = path.Item2, supplyInds = sinds, fn = max, drefs = drefs };
            //continue;

            //v = newSC[hdcmd.fn];

            if (supplyInds == null)
                throw new Exception("No changes to apply");
            foreach (int sid in supplyInds)
            {
                newSC.Add(sid, ese.Values[sid]);
            }

            if (path.Item1 != null)
            {
                foreach (int id in path.Item1)
                {
                    if (ese[id] is not CommandSE)
                    {
                        throw new Exception("Unreferenced solution path (should never see this)");
                    }
                    CommandSE cse = (CommandSE)ese[id];
                    newSC.Add(id, ProcessCmd(ese, newSC, cse));
                }
            }

            foreach ((int id, int iarg) in path.Item2)
            {
                if (ese[id] is not CommandSE)
                {
                    throw new Exception("Unreferenced inverse path (should never see this)");
                }
                CommandSE cse = (CommandSE)ese[id];
                v = InverseCmd(ese, newSC, cse, v, iarg);
                //newSC.Add(cse.cmdrefs[iarg].stackIndex, v);
            }
            
            runningExtCmds.Remove(cmname);
            return v;
        }

        private List<(int,int)> BuildDependencyTree(StackEntries sse, int query, SortedSet<int> knownInds, int firstStop = 0)
        {
            //list in this form:
            //refs 7 7->4,5,6  4->2,3 5->3,4 6->1,4
            // 7    7,4,5,6     7,4,2,3,5,3,4,6,1,4     7,4,2,3,5,3,2,3,6,1,2,3
            // 1    1,2,2,2     1,2,3,3,2,3,3,2,3,3     1,2,3,3,2,3,4,4,2,3,4,4
            //stop at ValueSE or knownInds or ind<firstStop
            if (firstStop < 0)
                throw new ArgumentOutOfRangeException("First stop must be >=0");
            List<(int,int)> tree = new List<(int,int)>() { (query, -1) };
            int dn = 0;
            bool done = false;
            while (!done)
            {
                done = true;
                int j = tree.Count;
                for (int i = 0; i < j; i++)
                {
                    (int l,int lv) = tree[i];
                    if (lv > 0)
                        continue;
                    tree[i] = (l, -lv);
                    if (l < firstStop || knownInds.Contains(l))
                    {
                        continue;
                    }
                    StackEntry se = sse[l];
                    if (se is CommandSE)
                    {
                        CommandSE cse = (CommandSE)se;
                        List<SRSingle>? rl;
                        if (cse is HDCommandSE)
                        {
                            //throw new Exception("Nested hypotheticals not supported");
                            rl = ((HDCommandSE)cse).drefs;
                        }
                        else
                        {
                            rl = cse.cmdrefs;
                        }
                        if (rl != null)
                        {
                            int pc = rl.Count;
                            if (pc > 0)
                            {
                                lv--;
                                for (int m = 0; m < pc; m++)
                                {
                                    i++;
                                    j++;
                                    tree.Insert(i, (rl[m].stackIndex, lv));
                                }
                                done = false;
                            }
                        }
                    }
                }

                dn++;
                if (dn > 1000)
                    throw new Exception("Too many iterations of dependency tree");
            }

            return tree;
        }

        private SortedSet<int> TrimAndCheck(ref List<(int,int)> tree, SortedSet<int> knownInds, int src = -1)
        {
            //remove deep entries where the ends are not in knownInds
            //and verify that all knownInds show up, with src (if src>0) showing up exactly once
            //and return other necessary indices (leaves)
            SortedSet<int> usedInds = new SortedSet<int>();
            int j = tree.Count;
            List<bool> keep = new List<bool>(j);
            for(int i=0; i<j; i++)
            {
                keep.Add(false);
                (int ind, int lv) = tree[i];
                if (lv <= 0)
                    throw new Exception("Tree not finalized (should never see this)");
                if (knownInds.Contains(ind))
                {
                    if (!usedInds.Add(ind) && ind == src)
                        throw new Exception("Dependency multiply valent");
                    keep[i] = true;
                    int r = i + 1;
                    while (r < j && tree[r].Item2 > lv)
                    {
                        r++;
                    }
                    r = r - (i + 1);
                    if (r > 0)
                    {
                        tree.RemoveRange(i + 1, r);
                        j -= r;
                    }
                    r = i;
                    int l = lv - 1;
                    while (r > 0)
                    {
                        r--;
                        if (tree[r].Item2 == l)
                        {
                            if (keep[r])
                                break;
                            keep[r] = true;
                            l--;
                        }
                    }
                }
            }
            if (!keep[0] || (usedInds.Count != knownInds.Count))
                throw new Exception("Dependency mismatch");
            
            usedInds.Clear();
            for (int i = 0; i < j; i++)
            {
                if (!keep[i])
                {
                    usedInds.Add(tree[i].Item1);
                    int lv = tree[i].Item2;
                    i++;
                    while (i < j && tree[i].Item2 > lv)
                    {
                        tree[i] = (tree[i].Item1,-tree[i].Item2);
                        i++;
                    }
                    i--;
                }
            }

            tree.RemoveAll((x) => (x.Item2 <= 0));

            return usedInds;
        }

        private List<int> BuildSolutionPath(List<(int,int)> tree)
        {
            //solve from deepest level to shallowest
            int j = tree.Count;
            int ml = 0;
            for (int i = 0; i < j; i++)
            {
                if (tree[i].Item2 > ml)
                    ml = tree[i].Item2;
                if (tree[i].Item2 <= 0)
                    throw new Exception("Tree not finalized (should never see this)");
            }
            SortedSet<int> evald = new SortedSet<int>();
            List<int> path = new List<int>();
            while (ml > 1)
            {
                ml--;
                for (int i = 0; i < (j - 1); i++)
                {
                    if (tree[i].Item2 == ml && tree[i + 1].Item2 > ml)
                    {
                        int n = tree[i].Item1;
                        if (evald.Add(n))
                        {
                            path.Add(n);
                        }
                    }
                }
            }

            return path;
        }

        private (List<int>,List<(int,int)>) BuildInversePath(List<(int, int)> tree, int src)
        {
            //first find forward path to prepare any inverse nodes
            //then find inverse nodes and argument number to solve for

            //locate source node and mark all parent nodes for inverse path
            //any sublevels not in inverse path should be solved forward
            //solve from deepest level to shallowest
            int j = tree.Count;
            int ml = 0;
            List<bool> keep = new List<bool>(j);
            for (int i = 0; i < j; i++)
            {
                keep.Add(false);
                (int ind, int lv) = tree[i];
                if (lv > ml)
                    ml = lv;
                if (lv <= 0)
                    throw new Exception("Tree not finalized (should never see this)");
                if (ind == src)
                {
                    keep[i] = true;
                    int r = i;
                    int l = lv - 1;
                    while (r > 0)
                    {
                        r--;
                        if (tree[r].Item2 == l)
                        {
                            if (keep[r])
                                throw new Exception("Source multiply valent (should never see this)");
                            keep[r] = true;
                            l--;
                        }
                    }
                }
            }
            if (!keep[0])
                throw new Exception("Source mismatch");

            SortedSet<int> evald = new SortedSet<int>();
            List<int> fwdpath = new List<int>();
            while (ml > 1)
            {
                ml--;
                for (int i = 0; i < (j - 1); i++)
                {
                    if (tree[i].Item2 == ml && (!keep[i]) && tree[i + 1].Item2 > ml)
                    {
                        int n = tree[i].Item1;
                        if (evald.Add(n))
                        {
                            fwdpath.Add(n);
                        }
                    }
                }
            }

            int? li = null;
            int ar = 0;
            ml = 1;
            List<(int, int)> invpath = new List<(int, int)>();
            //evald.Clear();

            for (int i = 0; i < j; i++)
            {
                if (keep[i])
                {
                    if (tree[i].Item2 != ml)
                        throw new Exception("Tree inversion error 1 (should never see this)");
                    if (li.HasValue)
                    {
                        if (!evald.Add(li.Value))
                        {
                            throw new Exception("Tree inversion error 3");
                        }
                        invpath.Add((li.Value, ar));
                    }
                    if (tree[i].Item1 == src)
                    {
                        break;
                    }
                    ml = tree[i].Item2 + 1;
                    li = tree[i].Item1;
                    ar = 0;
                }
                else
                {
                    if (tree[i].Item2 < ml)
                        throw new Exception("Tree inversion error 2 (should never see this)");
                    if (tree[i].Item2 == ml)
                        ar++;
                }
            }

            return (fwdpath, invpath);
        }
    }
}
