﻿namespace MCalcConsole
{
    internal class CmdProps
    {
        internal delegate double ComputeResult(List<double> args, int fnum, int arg);
        required public int minargs, maxargs;
        public int fnum = 0, outlines = 1;
        public bool inv = false;
        required public ComputeResult cmdfunc;
        public string? description;
    }

    internal static class Maths
    {
        public static readonly SortedList<string, CmdProps> mathCmds = new SortedList<string, CmdProps>
            {
                //arithmetic
                { "+", new CmdProps() { cmdfunc = Maths.Sum, inv = true, minargs = 2, maxargs = int.MaxValue, description = "Addition" } },
                { "++", new CmdProps() { cmdfunc = Maths.Sum, inv = true, fnum = -1, minargs = 1, maxargs = 1, description = "Add 1" } },
                { "--", new CmdProps() { cmdfunc = Maths.Sum, inv = true, fnum = -2, minargs = 1, maxargs = 1, description = "Subtract 1" } },
                { "*", new CmdProps() { cmdfunc = Maths.Product, inv = true, minargs = 2, maxargs = int.MaxValue, description = "Multiplication" } },
                { "|", new CmdProps() { cmdfunc = Maths.Parallel, inv = true, minargs = 2, maxargs = int.MaxValue, description = "Parallel combination 1/(1/x+1/y)" } },
                { "norm", new CmdProps() { cmdfunc = Maths.Norm, inv = true, minargs = 2, maxargs = int.MaxValue, description = "Norm (Euclidean distance)" } },
                { "mdist", new CmdProps() { cmdfunc = Maths.Norm, inv = true, fnum = 1, minargs = 2, maxargs = int.MaxValue, description = "Manhattan distance" } },
                { "-", new CmdProps() { cmdfunc = Maths.Subtract, inv = true, minargs = 2, maxargs = 2, description = "Subtraction" } },
                { "-<", new CmdProps() { cmdfunc = Maths.Subtract, inv = true, fnum = 2, minargs = 2, maxargs = 2, description = "Half-difference" } },
                { "-/", new CmdProps() { cmdfunc = Maths.Subtract, inv = true, fnum = 4, minargs = 2, maxargs = 2, description = "Difference as average ratio" } },
                { "~", new CmdProps() { cmdfunc = Maths.Negate, inv = true, minargs = 1, maxargs = 1, description = "Negation" } },
                { "abs", new CmdProps() { cmdfunc = Maths.Negate, fnum = 1, minargs = 1, maxargs = 1, description = "Absolute value" } },
                { "/", new CmdProps() { cmdfunc = Maths.Divide, inv = true, minargs = 2, maxargs = 2, description = "Division" } },
                { "\\", new CmdProps() { cmdfunc = Maths.IntPow, inv = true, fnum = -1, minargs = 1, maxargs = 1, description = "Inversion 1/Z" } },
                { "^", new CmdProps() { cmdfunc = Maths.Pow, fnum = 1, inv = true, minargs = 2, maxargs = 2, description = "Exponentiation" } },
                { "log", new CmdProps() { cmdfunc = Maths.Pow, fnum = -1, inv = true, minargs = 1, maxargs = 2, description = "Log base 10 (1 arg) or custom" } },
                { "logb", new CmdProps() { cmdfunc = Maths.Pow, fnum = -1, inv = true, minargs = 2, maxargs = 2, description = "Log custom base" } },
                { "sq", new CmdProps() { cmdfunc = Maths.IntPow, inv = true, fnum = 2, minargs = 1, maxargs = 1, description = "Square Z*Z" } },
                { "sqrt", new CmdProps() { cmdfunc = Maths.IntRt, inv = true, fnum = 2, minargs = 1, maxargs = 1, description = "Square root" } },
                { "cb", new CmdProps() { cmdfunc = Maths.IntPow, inv = true, fnum = 3, minargs = 1, maxargs = 1, description = "Cube" } },
                { "cbrt", new CmdProps() { cmdfunc = Maths.IntRt, inv = true, fnum = 3, minargs = 1, maxargs = 1, description = "Cube root" } },
                { "rt", new CmdProps() { cmdfunc = Maths.Root, inv = true, minargs = 2, maxargs = 2, description = "Zth root Y" } },
                { "pi", new CmdProps() { cmdfunc = Maths.Const, minargs = 0, maxargs = 0, description = "Pi constant" } },
                { "tau", new CmdProps() { cmdfunc = Maths.Const, fnum = 1, minargs = 0, maxargs = 0, description = "Tau constant (2*Pi)" } },
                { "e", new CmdProps() { cmdfunc = Maths.Const, fnum = 2, minargs = 0, maxargs = 0, description = "e constant" } },
                { ">", new CmdProps() { cmdfunc = Maths.Shift2, inv = true, fnum = 1, minargs = 1, maxargs = 1, description = "*2" } },
                { ">>", new CmdProps() { cmdfunc = Maths.Shift2, inv = true, fnum = 2, minargs = 1, maxargs = 1, description = "*4" } },
                { ">>>", new CmdProps() { cmdfunc = Maths.Shift2, inv = true, fnum = 3, minargs = 1, maxargs = 1, description = "*8" } },
                { ">>>>", new CmdProps() { cmdfunc = Maths.Shift2, inv = true, fnum = 4, minargs = 1, maxargs = 1, description = "*16" } },
                { "<", new CmdProps() { cmdfunc = Maths.Shift2, inv = true, fnum = -1, minargs = 1, maxargs = 1, description = "/2" } },
                { "<<", new CmdProps() { cmdfunc = Maths.Shift2, inv = true, fnum = -2, minargs = 1, maxargs = 1, description = "/4" } },
                { "<<<", new CmdProps() { cmdfunc = Maths.Shift2, inv = true, fnum = -3, minargs = 1, maxargs = 1, description = "/8" } },
                { "<<<<", new CmdProps() { cmdfunc = Maths.Shift2, inv = true, fnum = -4, minargs = 1, maxargs = 1, description = "/16" } },
                { "%", new CmdProps() { cmdfunc = Maths.Modulo, fnum = 0, minargs = 2, maxargs = 2, description = "Modulo" } },
                { "%i", new CmdProps() { cmdfunc = Maths.Modulo, fnum = 2, minargs = 2, maxargs = 2, description = "IEEE Remainder" } },
                { "ipart", new CmdProps() { cmdfunc = Maths.Modulo, fnum = 4, minargs = 1, maxargs = 1, description = "Integer part" } },
                { "fpart", new CmdProps() { cmdfunc = Maths.Modulo, fnum = 5, minargs = 1, maxargs = 1, description = "Fractional part" } },
                { "floor", new CmdProps() { cmdfunc = Maths.Round, fnum = -2, minargs = 1, maxargs = 1, description = "Next lower integer" } },
                { "ceil", new CmdProps() { cmdfunc = Maths.Round, fnum = -1, minargs = 1, maxargs = 1, description = "Next higher integer" } },
                { "round", new CmdProps() { cmdfunc = Maths.Round, fnum = 0, minargs = 1, maxargs = 1, description = "Nearest integer" } },
                { "round1", new CmdProps() { cmdfunc = Maths.Round, fnum = 1, minargs = 1, maxargs = 1, description = "Round to 1 significant digit" } },
                { "round2", new CmdProps() { cmdfunc = Maths.Round, fnum = 2, minargs = 1, maxargs = 1, description = "Round to 2 significant digits" } },
                { "round3", new CmdProps() { cmdfunc = Maths.Round, fnum = 3, minargs = 1, maxargs = 1, description = "Round to 3 significant digits" } },
                { "round4", new CmdProps() { cmdfunc = Maths.Round, fnum = 4, minargs = 1, maxargs = 1, description = "Round to 4 significant digits" } },
                { "mean", new CmdProps() { cmdfunc = Maths.Mean, inv = true, fnum = 0, minargs = 2, maxargs = int.MaxValue, description = "Arithmetic mean" } },
                { "+<", new CmdProps() { cmdfunc = Maths.Mean, inv = true, fnum = 0, minargs = 2, maxargs = 2, description = "Average" } },
                { "hmean", new CmdProps() { cmdfunc = Maths.Mean, inv = true, fnum = 1, minargs = 2, maxargs = int.MaxValue, description = "Harmonic mean" } },
                { "gmean", new CmdProps() { cmdfunc = Maths.Mean, inv = true, fnum = 2, minargs = 2, maxargs = int.MaxValue, description = "Geometric mean" } },
                { "std", new CmdProps() { cmdfunc = Maths.Mean, fnum = 3, minargs = 2, maxargs = int.MaxValue, description = "Standard deviation 1/N" } },
                { "sstd", new CmdProps() { cmdfunc = Maths.Mean, fnum = 4, minargs = 2, maxargs = int.MaxValue, description = "Standard deviation 1/(N-1)" } },
                { "s", new CmdProps() { cmdfunc = Maths.Trig, inv = true, fnum = 1, minargs = 1, maxargs = 1, description = "Sine" } },
                { "sd", new CmdProps() { cmdfunc = Maths.Trig, inv = true, fnum = 2, minargs = 1, maxargs = 1, description = "Sine of degrees" } },
                { "c", new CmdProps() { cmdfunc = Maths.Trig, inv = true, fnum = 3, minargs = 1, maxargs = 1, description = "Cosine" } },
                { "cd", new CmdProps() { cmdfunc = Maths.Trig, inv = true, fnum = 4, minargs = 1, maxargs = 1, description = "Cosine of degrees" } },
                { "t", new CmdProps() { cmdfunc = Maths.Trig, inv = true, fnum = 5, minargs = 1, maxargs = 1, description = "Tangent" } },
                { "td", new CmdProps() { cmdfunc = Maths.Trig, inv = true, fnum = 6, minargs = 1, maxargs = 1, description = "Tangent of degrees" } },
                { "sh", new CmdProps() { cmdfunc = Maths.Trig, inv = true, fnum = 7, minargs = 1, maxargs = 1, description = "Hyperbolic sine" } },
                { "ch", new CmdProps() { cmdfunc = Maths.Trig, inv = true, fnum = 8, minargs = 1, maxargs = 1, description = "Hyperbolic cosine" } },
                { "th", new CmdProps() { cmdfunc = Maths.Trig, inv = true, fnum = 9, minargs = 1, maxargs = 1, description = "Hyperbolic tangent" } },
                { "as", new CmdProps() { cmdfunc = Maths.Trig, inv = true, fnum = -1, minargs = 1, maxargs = 1, description = "Arcsine" } },
                { "asd", new CmdProps() { cmdfunc = Maths.Trig, inv = true, fnum = -2, minargs = 1, maxargs = 1, description = "Arcsine as degrees" } },
                { "ac", new CmdProps() { cmdfunc = Maths.Trig, inv = true, fnum = -3, minargs = 1, maxargs = 1, description = "Arccosine" } },
                { "acd", new CmdProps() { cmdfunc = Maths.Trig, inv = true, fnum = -4, minargs = 1, maxargs = 1, description = "Arccosine as degrees" } },
                { "at", new CmdProps() { cmdfunc = Maths.Trig, inv = true, fnum = -5, minargs = 1, maxargs = 2, description = "Arctangent" } },
                { "at2", new CmdProps() { cmdfunc = Maths.Trig, inv = true, fnum = -5, minargs = 2, maxargs = 2, description = "Arctangent" } },
                { "atd", new CmdProps() { cmdfunc = Maths.Trig, inv = true, fnum = -6, minargs = 1, maxargs = 2, description = "Arctangent as degrees" } },
                { "atd2", new CmdProps() { cmdfunc = Maths.Trig, inv = true, fnum = -6, minargs = 2, maxargs = 2, description = "Arctangent as degrees" } },
                { "ash", new CmdProps() { cmdfunc = Maths.Trig, inv = true, fnum = -7, minargs = 1, maxargs = 1, description = "Hyperbolic arcsine" } },
                { "ach", new CmdProps() { cmdfunc = Maths.Trig, inv = true, fnum = -8, minargs = 1, maxargs = 1, description = "Hyperbolic arccosine" } },
                { "ath", new CmdProps() { cmdfunc = Maths.Trig, inv = true, fnum = -9, minargs = 1, maxargs = 1, description = "Hyperbolic arctangent" } },
                { "exp", new CmdProps() { cmdfunc = Maths.Exp, inv = true, fnum = 1, minargs = 1, maxargs = 1, description = "e^Z" } },
                { "exp10", new CmdProps() { cmdfunc = Maths.Exp, inv = true, fnum = 2, minargs = 1, maxargs = 1, description = "10^Z" } },
                { "exp2", new CmdProps() { cmdfunc = Maths.Exp, inv = true, fnum = 3, minargs = 1, maxargs = 1, description = "2^Z" } },
                { "ln", new CmdProps() { cmdfunc = Maths.Exp, inv = true, fnum = -1, minargs = 1, maxargs = 1, description = "Log base e" } },
                { "log10", new CmdProps() { cmdfunc = Maths.Exp, inv = true, fnum = -2, minargs = 1, maxargs = 1, description = "Log base 10" } },
                { "log2", new CmdProps() { cmdfunc = Maths.Exp, inv = true, fnum = -3, minargs = 1, maxargs = 1, description = "Log base 2" } },
                { "gcd", new CmdProps() { cmdfunc = Maths.GCD, fnum = 0, minargs = 2, maxargs = int.MaxValue, description = "Greatest common divisor of integers" } },
                { "lcm", new CmdProps() { cmdfunc = Maths.GCD, fnum = 1, minargs = 2, maxargs = int.MaxValue, description = "Least common multiplier of integers" } },
                { "npr", new CmdProps() { cmdfunc = Maths.Comb, fnum = 0, minargs = 2, maxargs = 2, description = "Permutations" } },
                { "ncr", new CmdProps() { cmdfunc = Maths.Comb, fnum = 1, minargs = 2, maxargs = 2, description = "Combinations" } },
                { "factorial", new CmdProps() { cmdfunc = Maths.Comb, fnum = 2, minargs = 1, maxargs = 1, description = "Factorial" } },
                //multi-line commands:
                { "/%", new CmdProps() { cmdfunc = Maths.DivRem, fnum = 0, minargs = 2, maxargs = 2, outlines = 2, description = "Division with remainder" } },
                { "/%i", new CmdProps() { cmdfunc = Maths.DivRem, fnum = 2, minargs = 2, maxargs = 2, outlines = 2, description = "Division with IEEE remainder" } },
                { "parts", new CmdProps() { cmdfunc = Maths.DivRem, fnum = 4, minargs = 1, maxargs = 1, outlines = 2, description = "Integer and fractional parts" } },
                { "+-", new CmdProps() { cmdfunc = Maths.PMT, inv = true, fnum = 0, minargs = 2, maxargs = 2, outlines = 2, description = "Sum and difference" } },
                { "-+", new CmdProps() { cmdfunc = Maths.PMT, inv = true, fnum = 1, minargs = 2, maxargs = 2, outlines = 2, description = "Difference and sum" } },
                { "+-<", new CmdProps() { cmdfunc = Maths.PMT, inv = true, fnum = 3, minargs = 2, maxargs = 2, outlines = 2, description = "Half of Sum and difference" } },
                { "-+<", new CmdProps() { cmdfunc = Maths.PMT, inv = true, fnum = 4, minargs = 2, maxargs = 2, outlines = 2, description = "Half of Difference and sum" } },
                { "mstd", new CmdProps() { cmdfunc = Maths.Stats, fnum = 0, minargs = 2, maxargs = int.MaxValue, outlines = 2, description = "Mean and standard deviation 1/N" } },
                { "msstd", new CmdProps() { cmdfunc = Maths.Stats, fnum = 2, minargs = 2, maxargs = int.MaxValue, outlines = 2, description = "Mean and standard deviation 1/(N-1)" } },
                { "box", new CmdProps() { cmdfunc = Maths.Stats, fnum = 4, minargs = 2, maxargs = int.MaxValue, outlines = 2, description = "Enclosing range A+-B" } },
            };

        
        public static double Sum(List<double> args, int fnum, int arg) //2+ args
        {
            if (fnum == -1)
                return (arg < 0) ? (args[0] + 1) : (args[0] - 1);
            if (fnum == -2)
                return (arg < 0) ? (args[0] - 1) : (args[0] + 1);
            double s;
            if (arg < 0)
            {
                s = 0;
                foreach (double d in args)
                {
                    s += d;
                }
            }
            else
            {
                s = args[0];
                foreach (double d in args.Skip(1))
                {
                    s -= d;
                }
            }
            return s;
        }

        public static double Product(List<double> args, int fnum, int arg) //2+ args
        {
            double s;
            if(arg < 0)
            {
                s = 1;
                foreach (double d in args)
                {
                    s *= d;
                }
            }
            else
            {
                s = args[0];
                foreach (double d in args.Skip(1))
                {
                    s /= d;
                }
            }

            return s;
        }

        public static double Norm(List<double> args, int fnum, int arg) //1+ args
        {
            double s = 0;
            if (fnum == 1)
            {
                if (arg < 0)
                {
                    foreach (double d in args)
                    {
                        s += double.Abs(d);
                    }
                    return s;
                }
                else
                {
                    s = args[0];
                    foreach (double d in args.Skip(1))
                    {
                        s -= double.Abs(d);
                    }
                    return s;
                }
            }
            if (args.Count == 2)
            {
                return (arg < 0) ? double.Hypot(args[0], args[1]) : double.Sqrt(args[0] * args[0] - args[1] * args[1]);
            }
            if (arg < 0)
            {
                foreach (double d in args)
                {
                    s += d * d;
                }
                return double.Sqrt(s);
            }
            else
            {
                s = args[0] * args[0];
                foreach (double d in args.Skip(1))
                {
                    s -= d * d;
                }
                return double.Sqrt(s);
            }
        }

        public static double Parallel(List<double> args, int fnum, int arg)
        {
            double s;
            if (arg < 0)
            {
                s = 0;
                foreach (double d in args)
                {
                    s += (1.0 / d);
                }
                return 1.0 / s;
            }
            else
            {
                s = 1.0 / args[0];
                foreach (double d in args.Skip(1))
                {
                    s -= (1.0 / d);
                }
                return 1.0 / s;
            }
        }

        public static double Subtract(List<double> args, int fnum, int arg) //exactly 2 args
        {
            if (arg < 0)
            {
                switch (fnum)
                {
                    case 0:
                        return (args[0] - args[1]);
                    case 2:
                        return double.ScaleB(args[0] - args[1], -1);
                    case 4:
                        return (args[0] - args[1]) / double.ScaleB(args[0] + args[1], -1);
                }
            }
            else
            {
                switch (fnum)
                {
                    default:
                        return (arg == 0) ? (args[0] + args[1]) : (args[1] - args[0]);
                    case 2:
                        return (arg == 0) ? (double.ScaleB(args[0], 1) + args[1]) : (args[1] - double.ScaleB(args[0], 1));
                    case 4:
                        return (arg == 0) ? ((-args[1] * (args[0] + 2)) / (args[0] - 2)) : ((-args[1] * (args[0] - 2)) / (args[0] + 2));
                }
            }

            throw new Exception("Unknown fnum");
        }

        public static double Negate(List<double> args, int fnum, int arg) //exactly 1 args
        {
            if(fnum == 1)
                return double.Abs(args[0]);
            return -args[0];
        }

        public static double Divide(List<double> args, int fnum, int arg) //exactly 2 args
        {
            if (arg < 0)
                return args[0] / args[1];
            else
                return (arg == 0) ? (args[0] * args[1]) : (args[1] / args[0]);
        }

        public static double IntPow(List<double> args, int fnum, int arg) //exactly 1 args
        {
            return arg < 0 ? double.Pow(args[0], fnum) : double.RootN(args[0], fnum);
        }

        public static double IntRt(List<double> args, int fnum, int arg) //exactly 1 args
        {
            return arg < 0 ? double.RootN(args[0], fnum) : double.Pow(args[0], fnum);
        }

        public static double Pow(List<double> args, int fnum, int arg) //exactly 2 args
        {
            switch (fnum)
            {
                case 1:
                    if (args.Count < 2)
                        return (arg < 0) ? double.Exp10(args[0]) : double.Log10(args[0]);
                    else
                        if (arg < 0)
                            return double.Pow(args[0], args[1]);
                        else
                            return (arg == 0) ? double.Pow(args[0], 1.0 / args[1]) : double.Log(args[0], args[1]);
                case -1:
                    if (args.Count < 2)
                        return (arg < 0) ? double.Log10(args[0]) : double.Exp10(args[0]);
                    else
                        if (arg < 0)
                            return double.Log(args[0], args[1]);
                        else
                            return (arg == 0) ? double.Pow(args[1], args[0]) : double.Pow(args[1], 1.0 / args[0]);
            }
            throw new Exception("Unknown fnum");
        }

        public static double Exp(List<double> args, int fnum, int arg)
        {
            if (arg >= 0)
                fnum = -fnum;
            switch (fnum)
            {
                case 1:
                    return double.Exp(args[0]);
                case 2:
                    return double.Exp10(args[0]);
                case 3:
                    return double.Exp2(args[0]);
                case -1:
                    return double.Log(args[0]);
                case -2:
                    return double.Log10(args[0]);
                case -3:
                    return double.Log2(args[0]);
            }
            throw new Exception("Unknown fnum");
        }

        public static double Root(List<double> args, int fnum, int arg) //exactly 2 args
        {
            if (arg < 0)
                return double.IsInteger(args[1]) ? double.RootN(args[0], (int)args[1]) : double.Pow(args[0], 1.0 / args[1]);
            else
                return (arg == 0) ? double.Pow(args[0], args[1]) : double.Log(args[1], args[0]);
        }

        public static double Mean(List<double> args, int fnum, int arg) //1+ args
        {
            int nn = args.Count;
            double s = 0;
            switch (fnum)
            {
                case 0: //arithmetic
                    if (arg < 0)
                    {
                        foreach (double d in args)
                        {
                            s += d;
                        }
                        return s / nn;
                    }
                    else
                    {
                        s = args[0] * nn;
                        foreach (double d in args.Skip(1))
                        {
                            s -= d;
                        }
                        return s;
                    }
                case 1: //harmonic
                    if (arg < 0)
                    {
                        foreach (double d in args)
                        {
                            s += (1 / d);
                        }
                        return nn / s;
                    }
                    else
                    {
                        s = nn / args[0];
                        foreach (double d in args.Skip(1))
                        {
                            s -= (1 / d);
                        }
                        return 1 / s;
                    }
                case 2: //geometric
                    s = 0;
                    if (arg < 0)
                    {
                        foreach (double d in args)
                        {
                            s += double.Log(d);
                        }
                        return double.Exp(s / nn);
                    }
                    else
                    {
                        s = double.Log(args[0]) * nn;
                        foreach (double d in args.Skip(1))
                        {
                            s -= double.Log(d);
                        }
                        return double.Exp(s);
                    }
                case 3: //standard deviation
                case 4:
                    double s2 = 0, q = args[0], f = 0;
                    foreach (double d in args)
                    {
                        f = d - q;
                        s += f;
                        s2 += (f * f);
                    }
                    if(fnum == 3)
                    {
                        return (s2 - (s * s) / nn) / nn;
                    }
                    return (s2 - (s * s) / nn) / (nn - 1);
            }
            throw new Exception("Incorrect fnum");
        }

        public static double Const(List<double> args, int fnum, int arg) //0 args
        {
            switch (fnum)
            {
                case 0:
                    return double.Pi;
                case 1:
                    return double.Tau;
                case 2:
                    return double.E;
            }
            throw new Exception("Incorrect fnum");
        }

        public static double Shift2(List<double> args, int fnum, int arg) //1 args
        {
            return double.ScaleB(args[0], arg < 0 ? fnum : -fnum);
        }

        public static double Modulo(List<double> args, int fnum, int arg)
        {
            switch (fnum)
            {
                case 0:
                    return args[0] % args[1];
                case 1:
                    return args[1] % args[0];
                case 2:
                    return double.Ieee754Remainder(args[0], args[1]);
                case 3:
                    return double.Ieee754Remainder(args[1], args[0]);
                case 4:
                    return double.Truncate(args[0]);
                case 5:
                    return args[0] % 1;
            }
            throw new Exception("Incorrect fnum");
        }

        public static double Round(List<double> args, int fnum, int arg)
        {
            switch (fnum)
            {
                case -2:
                    return double.Floor(args[0]);
                case -1:
                    return double.Ceiling(args[0]);
                case 0:
                    return double.Round(args[0]);
            }
            if(fnum > 0 && fnum < 5)
            {
                if (!double.IsFinite(args[0]) || args[0] == 0)
                    return args[0];
                double a = (args[0] > 0) ? args[0] : -args[0];
                double b = double.Exp10(double.Ceiling(double.Log10(a)) - fnum);
                return double.Round(args[0] / b) * b;
            }
            throw new Exception("Incorrect fnum");
        }

        public static double Trig(List<double> args, int fnum, int arg)
        {
            if (arg >= 0)
                fnum = -fnum;
            switch (fnum)
            {
                case 1:
                    return double.Sin(args[0]);
                case 2:
                    return double.Sin(args[0] * (double.Pi / 180));
                case 3:
                    return double.Cos(args[0]);
                case 4:
                    return double.Cos(args[0] * (double.Pi / 180));
                case 5:
                    if (args.Count == 2 && arg >= 0)
                        return (arg == 0) ? (double.Tan(args[0]) * args[1]) : (args[1] / double.Tan(args[0]));
                    return double.Tan(args[0]);
                case 6:
                    if (args.Count == 2 && arg >= 0)
                        return (arg == 0) ? (double.Tan(args[0] * (double.Pi / 180)) * args[1]) : (args[1] / double.Tan(args[0] * (double.Pi / 180)));
                    return double.Tan(args[0] * (double.Pi / 180));
                case 7:
                    return double.Sinh(args[0]);
                case 8:
                    return double.Cosh(args[0]);
                case 9:
                    return double.Tanh(args[0]);
                case -1:
                    return double.Asin(args[0]);
                case -2:
                    return double.Asin(args[0]) * (180 / double.Pi);
                case -3:
                    return double.Acos(args[0]);
                case -4:
                    return double.Acos(args[0]) * (180 / double.Pi);
                case -5:
                    if (args.Count == 2)
                        return double.Atan2(args[0], args[1]);
                    return double.Atan(args[0]);
                case -6:
                    if (args.Count == 2)
                        return double.Atan2(args[0], args[1]) * (180 / double.Pi);
                    return double.Atan(args[0]) * (180 / double.Pi);
                case -7:
                    return double.Asinh(args[0]);
                case -8:
                    return double.Acosh(args[0]);
                case -9:
                    return double.Atanh(args[0]);
            }
            throw new Exception("Incorrect fnum");
        }

        public static double GCD(List<double> args, int fnum, int arg)
        {
            if (fnum == 0)
                return args.Aggregate(GCDi);
            else
                return args.Aggregate(LCMi);

            static double GCDi(double da, double db)
            {
                if (!double.IsInteger(da) || !double.IsInteger(db))
                    throw new ArgumentException("GCD only operates on integer arguments");
                int a = checked((int)da);
                int b = checked((int)db);
                if (a < 0) a = -a;
                if (b < 0) b = -b;
                while (a != 0 && b != 0)
                {
                    if (a > b)
                        a %= b;
                    else
                        b %= a;
                }
                return a | b;
            }

            static double LCMi(double da, double db)
            {
                double gcd = GCDi(da, db);
                if (gcd == 0)
                    return 0;
                return (da / gcd) * db;
            }
        }

        public static double Comb(List<double> args, int fnum, int arg)
        {
            switch (fnum)
            {
                case 0:
                    return frl(args[0], args[0] - args[1]);
                case 1:
                    if ((args[0] - args[1]) > args[1])
                        return frl(args[0], args[0] - args[1]) / frl(args[1], 1);
                    else
                        return frl(args[0], args[1]) / frl(args[0] - args[1], 1);
                case 2:
                    return frl(args[0], 1);
            }
            static double frl(double d, double f)
            {
                if (!double.IsInteger(d) || !double.IsInteger(f))
                    throw new ArgumentException("Factorial only operates on integer arguments");
                int q = checked((int)d);
                int w = checked((int)f);
                if (q == 0)
                    return 0;
                if (q < w || w < 0)
                    throw new ArgumentException("Factorial incorrect endpoint");
                if ((q - f) > 25)
                {
                    if(f < 2)
                        return Math.Sqrt(Math.Tau * d) * Math.Pow(d / Math.E, d); // https://en.wikipedia.org/wiki/Stirling%27s_approximation
                    return (Math.Sqrt(Math.Tau * d) * Math.Pow(d / Math.E, d)) / (Math.Sqrt(Math.Tau * f) * Math.Pow(f / Math.E, f));
                }
                q--;
                while (q > w)
                {
                    d = d * q;
                    q--;
                }
                return d;
            }
            throw new Exception("Incorrect fnum");
        }

        public static double[] Linf(List<double> args, int fnum) //(x1,y1,x2,y2) -> (a,b) such that y=a*x+b
        {
            return new double[] { (args[3] - args[1]) / (args[2] - args[0]), (args[1] * args[2] - args[3] * args[0]) / (args[2] - args[0]) };
        }

        /*public static double[] Eqs(List<double> args, int fnum)
        {
            switch (fnum)
            {
                case 1: //(a,b) such that y=a*x+b, return x such that y=0


            }
            return new double[] { (args[3] - args[1]) / (args[2] - args[0]), (args[1] * args[2] - args[3] * args[0]) / (args[2] - args[0]) };
        }*/

        public static double DivRem(List<double> args, int fnum, int arg) //(x,y)->integer divisor + remainder
        {
            double a = 0, b = 0;
            switch(fnum - (fnum % 2))
            {
                case 0:
                    a = double.Truncate(args[0] / args[1]);
                    b = args[0] - a * args[1];
                    break;
                case 2:
                    a = double.Round(args[0] / args[1]);
                    b = args[0] - a * args[1];
                    break;
                case 4:
                    a = double.Truncate(args[0]);
                    b = args[0] % 1;
                    break;
                default:
                    throw new Exception("Incorrect fnum");
            }
            return fnum % 2 == 0 ? a : b;
        }

        public static double PMT(List<double> args, int fnum, int arg) //exactly 2 args
        {
            if (arg < 0)
            {
                switch (fnum)
                {
                    case 0:
                        return args[0] + args[1];
                    case 1:
                        return args[0] - args[1];
                    case 2:
                        return args[0] + args[1];
                    case 3:
                        return (args[0] + args[1]) / 2;
                    case 4:
                        return (args[0] - args[1]) / 2;
                    case 5:
                        return (args[0] + args[1]) / 2;
                    default:
                        throw new Exception("Incorrect fnum");
                }
            }
            else
            {
                switch (fnum)
                {
                    case 0:
                        return args[0] - args[1];
                    case 1:
                        return (arg == 0) ? (args[0] + args[1]) : (args[1] - args[0]);
                    case 2:
                        return args[0] - args[1];
                    case 3:
                        return 2 * args[0] - args[1];
                    case 4:
                        return (arg == 0) ? (2 * args[0] + args[1]) : (args[1] - 2 * args[0]);
                    case 5:
                        return 2 * args[0] - args[1];
                    default:
                        throw new Exception("Incorrect fnum");
                }
            }
        }

        public static double Stats(List<double> args, int fnum, int arg) //1+ args
        {
            int nn = args.Count;
            double s = 0;
            double a, b;
            double s2 = 0, q = args[0], f = 0;
            if(fnum >= 4)
            {
                s = args[0];
                s2 = args[0];
                foreach (double d in args)
                {
                    if (d < s) s = d;
                    if (d > s2) s2 = d;
                }
                a = (s + s2) / 2;
                b = (s2 - s) / 2;
                return fnum == 4 ? a : b;
            }
            foreach (double d in args)
            {
                f = d - q;
                s += f;
                s2 += (f * f);
            }
            a = (s / nn) + q;
            if (fnum < 2)
            {
                b = (s2 - (s * s) / nn) / nn;
            }
            else
            {
                b = (s2 - (s * s) / nn) / (nn - 1);
            }
            return fnum % 2 == 0 ? a : b;
        }
    }
}
