﻿using System.Collections.ObjectModel;
using System.Text;

namespace MCalcConsole
{
    internal class StackEntries : ObservableCollection<StackEntry>
    {
        internal Stack<int> stackAbsRefInd = new Stack<int>(); //for visually renumbering the stack absolute references
        
        /// <summary>
        /// Returns absolute reference to entry relative to start of stack
        /// </summary>
        /// <param name="ai">1 for oldest entry, 2+ for newer entries</param>
        /// <returns>Single line reference</returns>
        /// <exception cref="ArgumentOutOfRangeException">Index outside stack bounds</exception>
        public SRSingle GetRefByAbsoluteIndex(int ai)
        {
            if(stackAbsRefInd.Count > 0)
                ai = ai + stackAbsRefInd.Peek();
            if (ai < 1 || ai > this.Count)
                throw new ArgumentOutOfRangeException("Specified absolute index does not exist in stack");
            return new SRSingle() { stackIndex = ai - 1 };
        }

        /// <summary>
        /// Returns absolute reference to entry relative to end of stack
        /// </summary>
        /// <param name="ai">1 for most recent entry, 2+ for older entries</param>
        /// <returns>Single line reference</returns>
        /// <exception cref="ArgumentOutOfRangeException">Index outside stack bounds</exception>
        public SRSingle GetRefByRelativeIndex(int ai)
        {
            if (ai < 1 || ai > this.Count)
                throw new ArgumentOutOfRangeException("Specified relative index does not exist in stack");
            return new SRSingle() { stackIndex = (this.Count - ai) };
        }

        public StackEntry GetByRef(SRSingle rf)
        {
            return this[rf.stackIndex];
        }

        public SRSingle GetLastCommand()
        {
            int i = this.Count - 1;
            while(i >= 0)
            {
                if (this[i] is CommandSE)
                    return new SRSingle() { stackIndex = i };
                i--;
            }
            throw new Exception("No command found in the stack");
        }

        public (SRSingle,SRSingle) GetLastIterator()
        {
            int i = this.Count - 1;
            int j = -1;
            while (i >= 0)
            {
                if (this[i] is CommandSE)
                {
                    if (((CommandSE)this[i]).mlin != 0)
                    {
                        if(j == -1)
                        {
                            j = i;
                        }
                    }
                    else
                    {
                        if (j != -1)
                        {
                            if (((CommandSE)this[i]).mlout == 0)
                            {
                                return (new SRSingle() { stackIndex = i }, new SRSingle() { stackIndex = j });
                            }
                        }
                    }
                }
                else
                {
                    if (j != -1)
                    {
                        throw new Exception("Incorrect iterator end in the stack (should never see this)");
                    }
                }
                i--;
            }
            throw new Exception("No iterator found in the stack");
        }

        public (SRSingle, SRSingle) GetLastMultiline()
        {
            int i = this.Count - 1;
            int j = -1;
            while (i >= 0)
            {
                if (this[i] is CommandSE)
                {
                    if (((CommandSE)this[i]).mlout != 0)
                    {
                        if (j == -1)
                        {
                            j = i;
                        }
                    }
                    else
                    {
                        if (j != -1)
                        {
                            return (new SRSingle() { stackIndex = i }, new SRSingle() { stackIndex = j });
                        }
                    }
                }
                else
                {
                    if (j != -1)
                    {
                        throw new Exception("Incorrect multiline end in the stack (should never see this)");
                    }
                }
                i--;
            }
            throw new Exception("No multiline found in the stack");
        }

        public SRSingle GetLastValue()
        {
            int i = this.Count - 1;
            while (i >= 0)
            {
                if (this[i] is ValueSE)
                    return new SRSingle() { stackIndex = i };
                i--;
            }
            throw new Exception("No value found in the stack");
        }

        public SRSingle GetLastAdjValue()
        {
            int i = this.Count - 1;
            int j = -1;
            while (i >= 0)
            {
                if (this[i] is ValueSE)
                    j = i;
                else
                    if (j != -1)
                        break;
                i--;
            }
            if(j != -1)
                return new SRSingle() { stackIndex = j };
            throw new Exception("No adj value found in the stack");
        }

        public void AddToStack(StackEntry rse)
        {
            int j = this.Count + 1;
            this.Add(rse);
            if (rse is RenumberSE)
            {

                stackAbsRefInd.Push(((RenumberSE)rse).newStart);
            }
            else
            {
                j -= (stackAbsRefInd.Count == 0) ? 0 : stackAbsRefInd.Peek();
                rse.AR = Convert(j);
            }
        }

        internal string Convert(int index)
        {
            if (index < 1)
                return "?";
            //if (index > MaxIndex)
            //    return "";
            if (index < 26)
            {
                return ((char)('Z' - (index - 1))).ToString();
            }
            else
            {
                StringBuilder sb = new StringBuilder(3);
                do
                {
                    sb.Append('A');
                    index -= 25;
                } while (index >= 26);
                sb.Append((char)('Z' - (index - 1)));
                return sb.ToString();
            }
        }
    }

    internal abstract class StackReference
    {
        
    }

    internal class SRSingle : StackReference
    {
        public int stackIndex;
    }

    internal class SRIterator : StackReference
    {
        public List<SRSingle> refs = new List<SRSingle>();
    }

    internal class VirtualStack
    {
        public SortedList<int, SCV> Stack = new SortedList<int, SCV>();
        public struct SCV { public double Value; public bool Initial; };
        public VirtualStack CopyInit()
        {
            VirtualStack vs = new VirtualStack();
            foreach (KeyValuePair<int, SCV> kvp in Stack)
            {
                if (kvp.Value.Initial)
                    vs.Stack.Add(kvp.Key, kvp.Value);
            }
            return vs;
        }
    }
}
