﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Runtime.InteropServices;
using System.Text;
using System.Threading.Tasks;
using System.Windows;
using System.Windows.Controls;
using System.Windows.Data;
using System.Windows.Documents;
using System.Windows.Input;
using System.Windows.Media;
using System.Windows.Media.Imaging;
using System.Windows.Shapes;
using FFTWSharp;

namespace ECGDisplay
{
    /// <summary>
    /// Interaction logic for SleepTracker.xaml
    /// </summary>
    public partial class SleepTracker : Window, IBCIListener
    {
        public bool TrackerEnabled { get; set; }

        string fndata = System.IO.Directory.GetCurrentDirectory() + "\\DATA_" + DateTime.Now.ToString("yyyy-MM-dd_HH-mm") + ".txt";
        long ticks_start = DateTime.Now.Ticks;

        double[] fft_data_in; //data to pass to FFT
        double[] fftd_data; //data after FFT has been called, complex format [re][im]
        double[] fftd_amplitudes; //amplitude only of FFT data
        double[] fftd_frequencies; //frequencies corresponding to amplitudes
        int data_in_ctr = 0;
        const int seconds = 4;
        const int dataRate = 250;
        const int numPts = seconds * dataRate;
        int analyzed_num_ctr = 0;
        const int analysis_write_at = 60 / seconds; //write out data once a minute

        double[] hr_data_in1, hr_data_in2;
        bool hrOver = false;
        double[] temp_data_in;

        double[] rem_values, temp_values, accel_values;
        int[] hr_values;

        //float lastX = 0, lastY = 0, lastZ = 0, lastSummed = 0;
        double lastSummed = 0, lastTheta = 0, lastSide = 0;

        GCHandle fft_in, fft_out; //when reserving memory for FFT, ensure it will be unreserved to avoid memory eating
        IntPtr fftPlan;
        bool destroyHandles = false;

        public SleepTracker()
        {
            InitializeComponent();

            //FFT wrapper and example: Github tszalay/FFTWSharp
            //Importing wisdom (wisdom speeds up the plan creation process, if that plan was previously created at least once)
            //fftwf.import_wisdom_from_filename("wisdom.wsd");

            hr_data_in1 = new double[numPts / 2];
            hr_data_in2 = new double[numPts / 2];
            temp_data_in = new double[numPts];

            rem_values = new double[analysis_write_at];
            hr_values = new int[analysis_write_at];
            temp_values = new double[analysis_write_at];
            accel_values = new double[analysis_write_at];

            fft_data_in = new double[numPts];
            fftd_data = new double[numPts + 2];
            int pl = (numPts + 2) / 2;
            fftd_frequencies = new double[pl];

            //max frequency is dataRate / 2, min frequency is 1 / seconds in sample
            double df = 1.0 / seconds;

            for (int f = 0; f < pl; f++)
            {
                fftd_frequencies[f] = f * df;
            }
            fftd_amplitudes = new double[pl];

            fft_in = GCHandle.Alloc(fft_data_in, GCHandleType.Pinned); //must be freed later
            fft_out = GCHandle.Alloc(fftd_data, GCHandleType.Pinned); //must be freed later
            fftPlan = fftw.dft_r2c_1d(numPts, fft_in.AddrOfPinnedObject(), fft_out.AddrOfPinnedObject(), fftw_flags.Estimate);
            destroyHandles = true;

            TrackerEnabled = false;

            this.DataContext = this;
        }

        ~SleepTracker()
        {
            if (destroyHandles)
            {
                //fftw.export_wisdom_to_filename("wisdom.wsd");
                fftw.destroy_plan(fftPlan);
                fft_in.Free();
                fft_out.Free();
            }
        }

        public void Initialize()
        {
            this.Show();
        }

        public void Shutdown()
        {
            if (destroyHandles)
            {
                //fftw.export_wisdom_to_filename("wisdom.wsd");
                fftw.destroy_plan(fftPlan);
                fft_in.Free();
                fft_out.Free();
                destroyHandles = false;
            }

            this.Close();
        }

        public void AnalyzePacket(BCIPacket pk)
        {
            if (!TrackerEnabled)
                return;

            //if(pk.accelX != null)
            //{
                //With board flat / LED pointing up, X = 0, Y = 0, Z = 1
                //With channel inputs pointing up, Y = 1
                //With board switch pointing up, X = 1
                double sqr = Math.Sqrt(pk.accelX * pk.accelX + pk.accelY * pk.accelY);
                double theta = Math.Atan2(pk.accelZ, sqr); //this gives pi/2 for laying on back, 0 for laying on side, -pi/2 for stomach
                double sideAngle = Math.Atan2(pk.accelX, pk.accelZ); //this gives 0 for back, pi/2 for right side, -pi/2 for left side, +-pi for stomach depending on left/right approach
                /*lastSummed += Math.Abs(lastX - pk.accelX);
                lastSummed += Math.Abs(lastY - pk.accelY);
                lastSummed += Math.Abs(lastZ - pk.accelZ);
                lastX = pk.accelX;
                lastY = pk.accelY;
                lastZ = pk.accelZ;*/
                lastSummed += Math.Abs(lastTheta - theta);
                lastTheta = theta;
                lastSide = sideAngle;
            //}

            foreach (BCIDataPoint bd in pk.sequenceData)
            {
                if (bd.channelNumber == BCIChannel.Number.c3)
                {
                    //track heart rate
                    if(data_in_ctr < numPts / 2)
                    {
                        hr_data_in1[data_in_ctr] = bd.value;
                    }
                    else
                    {
                        hr_data_in2[data_in_ctr - numPts / 2] = bd.value;
                    }   
                }
                if (bd.channelNumber == BCIChannel.Number.c4)
                {
                    //track REM
                    fft_data_in[data_in_ctr] = bd.value;
                }
                if (bd.channelNumber == BCIChannel.Number.c5)
                {
                    //track temperature
                    temp_data_in[data_in_ctr] = bd.value;
                }
            }

            data_in_ctr++;

            if(data_in_ctr == numPts)
            {
                doAnalysis();
                data_in_ctr = 0;
            }
        }

        private void doAnalysis()
        {
            accel_values[analyzed_num_ctr] = lastSummed * lastSummed; //accentuate high motion levels
            lastSummed = 0;
            
            //do FFT for REM data
            fftw.execute(fftPlan); //do FFT transform from data_in to fftd_data
            for (int f = 0; f < fftd_amplitudes.Length; f++)
            {
                fftd_amplitudes[f] = Math.Sqrt(fftd_data[f * 2] * fftd_data[f * 2] + fftd_data[f * 2 + 1] * fftd_data[f * 2 + 1]) * 2.0 / (fftd_data.Length - 2); //amplitude of FFT (element 0 and n need to be not multiplied by 2 for accurate amplitudes due to folding of positive and negative spectrum here)
            }
            fftd_amplitudes[0] *= 0.5;
            fftd_amplitudes[fftd_amplitudes.Length - 1] *= 0.5;

            double remCombined = 0;
            for (int f=1; f<13; f++) //sum up amplitudes from 0.25 to 3.0 Hz
            {
                remCombined += fftd_amplitudes[f];
            }
            remCombined *= 10000;
            remCombined *= remCombined;

            //do min/max counter for HR data
            double min = hr_data_in1.Min();
            double max = hr_data_in1.Max();
            double thr1 = min + (max - min) * 0.4;
            double thr2 = min + (max - min) * 0.6;
            int beats = 0;
            for(int f=0; f<hr_data_in1.Length; f++) //hysteresis switch type counter
            {
                if (hrOver)
                {
                    if (hr_data_in1[f] < thr1)
                        hrOver = false;
                }
                else
                {
                    if(hr_data_in1[f] > thr2)
                    {
                        hrOver = true;
                        beats++;
                    }
                }
            }
            min = hr_data_in2.Min();
            max = hr_data_in2.Max();
            thr1 = min + (max - min) * 0.4;
            thr2 = min + (max - min) * 0.6;
            for (int f = 0; f < hr_data_in2.Length; f++) //hysteresis switch type counter
            {
                if (hrOver)
                {
                    if (hr_data_in2[f] < thr1)
                        hrOver = false;
                }
                else
                {
                    if (hr_data_in2[f] > thr2)
                    {
                        hrOver = true;
                        beats++;
                    }
                }
            }


            //do simple averaging for temperature data
            double tavg = temp_data_in.Average();

            rem_values[analyzed_num_ctr] = remCombined;
            hr_values[analyzed_num_ctr] = beats;
            temp_values[analyzed_num_ctr] = tavg;

            displayAnalysisResults();

            analyzed_num_ctr++;

            if(analyzed_num_ctr >= analysis_write_at)
            {
                writeAnalysisResults();
                analyzed_num_ctr = 0;
            }
        }

        private void displayAnalysisResults()
        {
            StringBuilder sb = new StringBuilder(DateTime.Now.ToString("HH:mm:ss "));
            sb.Append("rem=");
            sb.Append(rem_values[analyzed_num_ctr]);
            sb.Append("  HR=");
            sb.Append(hr_values[analyzed_num_ctr]);
            sb.Append("  acc=");
            sb.Append(accel_values[analyzed_num_ctr]);
            sb.Append("  tht=");
            sb.Append(lastTheta);
            sb.Append("  sid=");
            sb.Append(lastSide);
            sb.Append("  temp=");
            sb.Append(temp_values[analyzed_num_ctr]);
            this.Dispatcher.Invoke(() => { AnalysisOnceOutput.Text = sb.ToString(); });
        }
        
        private void writeAnalysisResults()
        {
            double remTotal = Math.Sqrt(rem_values.Sum());
            int hrTotal = hr_values.Sum();
            double tempAvg = temp_values.Average();
            double accTotal = accel_values.Sum();

            StringBuilder sb = new StringBuilder(DateTime.Now.ToString("dd\tHH:mm:ss\t"));
            long dt = DateTime.Now.Ticks - ticks_start;
            sb.Append(dt / 10000000);
            sb.Append("\t");
            sb.Append(remTotal);
            sb.Append("\t");
            sb.Append(hrTotal);
            sb.Append("\t");
            sb.Append(accTotal);
            sb.Append("\t");
            sb.Append(lastTheta);
            sb.Append("\t");
            sb.Append(lastSide);
            sb.Append("\t");
            sb.Append(tempAvg);
            sb.AppendLine();

            System.IO.File.AppendAllText(fndata, sb.ToString());

            this.Dispatcher.Invoke(() => { AnalysisOutput.Text = sb.ToString(); });
        }

        private void FileButton_Click(object sender, RoutedEventArgs e)
        {
            if(OutputFileSelect("Sleep Study " + DateTime.Now.ToString("yyyy-MM-dd") + ".txt"))
            {
                StringBuilder sb = new StringBuilder("Day\tTime\t");
                sb.Append("sec=");
                sb.Append("\t");
                sb.Append("rem=");
                sb.Append("\t");
                sb.Append("HR=");
                sb.Append("\t");
                sb.Append("acc=");
                sb.Append("\t");
                sb.Append("tht=");
                sb.Append("\t");
                sb.Append("sid=");
                sb.Append("\t");
                sb.Append("temp=");
                sb.AppendLine();

                System.IO.File.AppendAllText(fndata, sb.ToString());

                ticks_start = DateTime.Now.Ticks;
            }
        }

        bool OutputFileSelect(string defaultName)
        {
            Microsoft.Win32.SaveFileDialog dlg = new Microsoft.Win32.SaveFileDialog();
            dlg.DefaultExt = ".txt";
            dlg.FileName = defaultName;
            dlg.Filter = "TXT Files (.txt)|*.txt";
            dlg.InitialDirectory = Environment.GetFolderPath(Environment.SpecialFolder.MyDocuments);

            if (dlg.ShowDialog() == true)
            {
                fndata = dlg.FileName;
                return true;
            }

            return false;
        }
    }
}
