﻿using System;
using System.Collections.Generic;
using System.IO.Ports;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

namespace ECGDisplay
{
    public class InterfaceBCISerial
    {
        public delegate void DataReady(BCIPacket data); //this gets called when serial port has received data
        public bool IsStreaming { get { return isStreaming; } }

        SerialPort comPort = null;
        byte[] serialBuffer = new byte[40];
        int[] auxPrepBuffer = new int[10];
        int[] auxReadyBuffer = new int[10];
        DataReady drExternal;
        bool isStreaming = false;
        bool wasLastPacketValid = false;

        BCIBoard myBoard;

        //counters
        public int uncorrectablePacketsCounter = 0, correctedPacketsCounter = 0, correctPacketsCounter = 0, missedPacketsCounter = 0, correctedBitsCounter = 0, uncorrectableBitsCounter = 0;
        byte lastPacketNumber = 255;
        double lastPacketTime = 0;
        const double packet_dt = 1.0 / 250.0; //seconds between packets

        //Error correction / Hamming code
        readonly byte[,] pargen = new byte[,]{
              { 0, 1, 3, 4, 6, 8,10,11,13,15,17,19,21,23,25},
              { 0, 2, 3, 5, 6, 9,10,12,13,16,17,20,21,24,25},
              { 1, 2, 3, 7, 8, 9,10,14,15,16,17,22,23,24,25},
              { 4, 5, 6, 7, 8, 9,10,18,19,20,21,22,23,24,25},
              {11,12,13,14,15,16,17,18,19,20,21,22,23,24,25}
            };
        readonly int[] parErrorMap = new int[] { -1, 25, 26, 0, 27, 1, 2, 3, 28, 4, 5, 6, 7, 8, 9, 10, 29, 11, 12, 13, 14, 15, 16, 17, 18, 19, 20, 21, 22, 23, 24, -2 };
        private enum PacketErrorStatus { Correct, Corrected, NotCorrectable };

        public InterfaceBCISerial(BCIBoard board, string port, DataReady dataReady)
        {
            myBoard = board;
            comPort = new SerialPort(port, 115200, Parity.None, 8, StopBits.One);
            comPort.NewLine = "\r\n"; //necessary line terminator
            drExternal = dataReady; //this will get called by our event handler when there is actually data available
            comPort.DataReceived += ComPort_DataReceived;
            comPort.Open();
        }
        
        private void ComPort_DataReceived(object sender, SerialDataReceivedEventArgs e)
        {
            if (!isStreaming)
            {
                while (comPort.BytesToRead > 0) //dump all buffer
                    comPort.ReadByte();
                return;
            }

            while (comPort.BytesToRead >= 33)
            {
                int ch;
                do
                {
                    if (comPort.BytesToRead < 33)
                        return;
                    ch = comPort.ReadByte();
                } while (ch != 0xA0); //align to start byte 0xA0

                comPort.Read(serialBuffer, 0, 32);

                //byte 0-23 = 24 bits per channel * 8 channels, signed int MSB first
                //byte 24 = aux data transmission 0xYZ
                //          Y = 0-F increment for each packet, where even means sequence 0-7 and odd means sequence 8-15, and also what Z is
                //          Z = 4 bits of aux array, left to right, giving 8 bytes [LONB, LOPB, LOND, LOPD, GPIO, X, Y, Z]
                //bytes 25-30 = 5 parity bytes and 1 total parity byte
                //31 = footer 0xC* where * = 1 for 8ch error protection mode and 8 for 10ch mode

                if (serialBuffer[31] == 0xC1)
                {
                    PacketErrorStatus pe = errorCorrectionAlgs();

                    if (pe == PacketErrorStatus.NotCorrectable)
                    {
                        wasLastPacketValid = false;
                        break;
                    }

                    lastPacketNumber++;
                    int recdNumber = (serialBuffer[24] & 0xF0) >> 4;
                    while (recdNumber != (lastPacketNumber & 0x0F))
                    {
                        wasLastPacketValid = false;
                        lastPacketNumber++;
                        lastPacketTime += packet_dt;
                        missedPacketsCounter++;
                    } //this handles out of order packets by not showing them in the plot while keeping time basis consistent

                    lastPacketTime += packet_dt; //for the code below this is the new time
                                                 //extract aux bytes state
                    extractAuxByte(serialBuffer[24], wasLastPacketValid);
                    //process data into a nice format
                    BCIPacket pk = DecodeSerialPacket8(serialBuffer, lastPacketTime);
                    //send outside for display and analysis
                    drExternal(pk);
                    wasLastPacketValid = true;
                }
                else
                {
                    if (serialBuffer[31] == 0xC8)
                    {
                        lastPacketNumber++;
                        int recdNumber = (serialBuffer[30] & 0xF0) >> 4;
                        while (recdNumber != (lastPacketNumber & 0x0F))
                        {
                            wasLastPacketValid = false;
                            lastPacketNumber++;
                            lastPacketTime += packet_dt;
                            missedPacketsCounter++;
                        } //this handles out of order packets by not showing them in the plot while keeping time basis consistent

                        lastPacketTime += packet_dt; //for the code below this is the new time
                                                     //extract aux bytes state
                        extractAuxByte(serialBuffer[30], wasLastPacketValid);
                        //process data into a nice format
                        BCIPacket pk = DecodeSerialPacket10(serialBuffer, lastPacketTime);
                        //send outside for display and analysis
                        drExternal(pk);
                        wasLastPacketValid = true;
                    }
                    else
                    {
                        uncorrectablePacketsCounter++;
                    }
                }
            }
        }

        private void extractAuxByte(byte auxByte, bool lastValid)
        {
            //extract data into local aux array
            int ctr = (auxByte & 0xF0) >> 4;
            int data = auxByte & 0x0F;
            int bn = ctr >> 1;
            if ((ctr & 0x01) != 0)
            { //this is an odd number so we are finishing up the second half of byte
                if (!lastValid)
                    return; //don't want half of a wrong byte
                auxPrepBuffer[bn] |= data;
                auxReadyBuffer[bn] = auxPrepBuffer[bn];
            }
            else
            { //this is an even number so only first half of byte
                auxPrepBuffer[bn] = data << 4;
            }
        }

        private PacketErrorStatus errorCorrectionAlgs()
        {
            byte[] parity_calc = new byte[6] { 0, 0, 0, 0, 0, 0 };
            //get 5 Hamming bytes
            for(int p=0; p<5; p++)
            {
                for (int i = 0; i < 14; i++)
                { //there is no data in byte 25 so only go up to index 13
                    parity_calc[p] ^= serialBuffer[pargen[p,i]];
                }
            }
            //and the total parity
            for (int i = 0; i < 30; i++)
            {
                parity_calc[5] ^= serialBuffer[i];
            }
            //now compare them with what we actually got
            bool allCorrect = true;
            for (int i = 0; i < 6; i++)
            {
                parity_calc[i] ^= serialBuffer[25+i];
                if(parity_calc[i] != 0)
                {
                    allCorrect = false;
                }
            }

            if (allCorrect)
            {
                correctPacketsCounter++;
                return PacketErrorStatus.Correct;
            }
            //need to find whether it is correctable
            //this means that:
            //  Where Hamming bits are zero, total parity is zero or one (if one, the total parity is assumed wrong)
            //  Where Hamming bits are nonzero, total parity is one (if one, it is a correctable 1-bit error, if zero, it is not correctable)
            //  Where the error is correctable, its location is not position 26 since that is never used in the calculation
            int[] correctLocs = new int[8];
            correctLocs[0] = ((parity_calc[0] & 0x01)) | ((parity_calc[1] & 0x01) << 1) | ((parity_calc[2] & 0x01) << 2) | ((parity_calc[3] & 0x01) << 3) | ((parity_calc[4] & 0x01) << 4);
            correctLocs[1] = ((parity_calc[0] & 0x02) >> 1) | ((parity_calc[1] & 0x02)) | ((parity_calc[2] & 0x02) << 1) | ((parity_calc[3] & 0x02) << 2) | ((parity_calc[4] & 0x02) << 3);
            correctLocs[2] = ((parity_calc[0] & 0x04) >> 2) | ((parity_calc[1] & 0x04) >> 1) | ((parity_calc[2] & 0x04)) | ((parity_calc[3] & 0x04) << 1) | ((parity_calc[4] & 0x04) << 2);
            correctLocs[3] = ((parity_calc[0] & 0x08) >> 3) | ((parity_calc[1] & 0x08) >> 2) | ((parity_calc[2] & 0x08) >> 1) | ((parity_calc[3] & 0x08)) | ((parity_calc[4] & 0x08) << 1);
            correctLocs[4] = ((parity_calc[0] & 0x10) >> 4) | ((parity_calc[1] & 0x10) >> 3) | ((parity_calc[2] & 0x10) >> 2) | ((parity_calc[3] & 0x10) >> 1) | ((parity_calc[4] & 0x10));
            correctLocs[5] = ((parity_calc[0] & 0x20) >> 5) | ((parity_calc[1] & 0x20) >> 4) | ((parity_calc[2] & 0x20) >> 3) | ((parity_calc[3] & 0x20) >> 2) | ((parity_calc[4] & 0x20) >> 1);
            correctLocs[6] = ((parity_calc[0] & 0x40) >> 6) | ((parity_calc[1] & 0x40) >> 5) | ((parity_calc[2] & 0x40) >> 4) | ((parity_calc[3] & 0x40) >> 3) | ((parity_calc[4] & 0x40) >> 2);
            correctLocs[7] = ((parity_calc[0] & 0x80) >> 7) | ((parity_calc[1] & 0x80) >> 6) | ((parity_calc[2] & 0x80) >> 5) | ((parity_calc[3] & 0x80) >> 4) | ((parity_calc[4] & 0x80) >> 3);

            byte mask = 0x01;
            bool isCorrectable = true;
            for(int bit=0; bit<8; bit++)
            {
                if (correctLocs[bit] != 0)
                {
                    if((parity_calc[5] & mask) == 0 || correctLocs[bit] == 31)
                    {
                        isCorrectable = false;
                        uncorrectableBitsCounter++;
                    }
                    else
                    {
                        serialBuffer[parErrorMap[correctLocs[bit]]] ^= mask;
                        correctedBitsCounter++;
                    }
                }
                mask <<= 1;
            }
            if (!isCorrectable)
            {
                uncorrectablePacketsCounter++;
                return PacketErrorStatus.NotCorrectable;
            }
            else
            {
                correctedPacketsCounter++;
                return PacketErrorStatus.Corrected;
            }
        }

        BCIPacket DecodeSerialPacket8(byte[] serialBytes, double startTime)
        {
            BCIPacket pk = new BCIPacket();
            pk.packetTime = startTime;
            pk.IOData = (byte)auxReadyBuffer[4];
            int loffP = (byte)auxReadyBuffer[0];
            int loffN = (byte)auxReadyBuffer[1];
            int loffP_daisy = (byte)auxReadyBuffer[2];
            int loffN_daisy = (byte)auxReadyBuffer[3];
            sbyte ah = (sbyte)auxReadyBuffer[5];
            pk.accelX = ah / 64.0f; //convert to units of [g] given +-2g scale
            ah = (sbyte)auxReadyBuffer[6];
            pk.accelY = ah / 64.0f;
            ah = (sbyte)auxReadyBuffer[7];
            pk.accelZ = ah / 64.0f;
            pk.sequenceData = new BCIDataPoint[8];

            int seqStart = (serialBytes[25] >> 1) & 0x08; //start point of sequence in this packet - either channel 0 for even ctr or 8 for odd ctr

            for (int c = 0; c < 8; c++)
            {
                int chan = (int)myBoard.channelSequence[seqStart + c];
                BCIChannel bch = myBoard.BoardChannels[chan - 1];
                int d = c * 3;
                BCIDataPoint dat = new BCIDataPoint(bch, serialBytes[d], serialBytes[d + 1], serialBytes[d + 2], startTime);
                if (chan < 9)
                {
                    int c2 = chan - 1;
                    dat.LOffP = (loffP & (0x01 << c2)) != 0;
                    dat.LOffN = (loffN & (0x01 << c2)) != 0;
                }
                else
                {
                    int c2 = chan - 9;
                    dat.LOffP = (loffP_daisy & (0x01 << c2)) != 0;
                    dat.LOffN = (loffN_daisy & (0x01 << c2)) != 0;
                }
                pk.sequenceData[c] = dat;
            }

            double time_inc = 0; //unwrap the interpolation allowed by sequence
            switch (myBoard.dataRate)
            {
                case BCIBoard.DataRate.x250:
                    //time_inc = 0;
                    break;
                case BCIBoard.DataRate.x500:
                    time_inc = (1.0 / 500.0);
                    for (int c = 4; c < 8; c++)
                    {
                        pk.sequenceData[c].time += time_inc;
                    }
                    break;
                case BCIBoard.DataRate.x1000:
                    time_inc = (1.0 / 1000.0);
                    for (int c = 2; c < 8; c+=2)
                    {
                        pk.sequenceData[c].time += time_inc;
                        pk.sequenceData[c+1].time += time_inc;
                        time_inc += (1.0 / 1000.0);
                    }
                    break;
                case BCIBoard.DataRate.x2000:
                    time_inc = (1.0 / 2000.0);
                    for (int c = 1; c < 8; c ++)
                    {
                        pk.sequenceData[c].time += time_inc;
                        time_inc += (1.0 / 2000.0);
                    }
                    break;
            }

            return pk;
        }

        BCIPacket DecodeSerialPacket10(byte[] serialBytes, double startTime)
        {
            BCIPacket pk = new BCIPacket();
            pk.packetTime = startTime;
            pk.IOData = (byte)auxReadyBuffer[4];
            int loffP = (byte)auxReadyBuffer[0];
            int loffN = (byte)auxReadyBuffer[1];
            int loffP_daisy = (byte)auxReadyBuffer[2];
            int loffN_daisy = (byte)auxReadyBuffer[3];
            sbyte ah = (sbyte)auxReadyBuffer[5];
            pk.accelX = ah / 64.0f; //convert to units of [g] given +-2g scale
            ah = (sbyte)auxReadyBuffer[6];
            pk.accelY = ah / 64.0f;
            ah = (sbyte)auxReadyBuffer[7];
            pk.accelZ = ah / 64.0f;
            pk.sequenceData = new BCIDataPoint[10];

            for (int c = 0; c < 10; c++)
            {
                int chan = (int)myBoard.channelSequence[c];
                BCIChannel bch = myBoard.BoardChannels[chan - 1];
                int d = c * 3;
                BCIDataPoint dat = new BCIDataPoint(bch, serialBytes[d], serialBytes[d + 1], serialBytes[d + 2], startTime);
                if (chan < 9)
                {
                    int c2 = chan - 1;
                    dat.LOffP = (loffP & (0x01 << c2)) != 0;
                    dat.LOffN = (loffN & (0x01 << c2)) != 0;
                }
                else
                {
                    int c2 = chan - 9;
                    dat.LOffP = (loffP_daisy & (0x01 << c2)) != 0;
                    dat.LOffN = (loffN_daisy & (0x01 << c2)) != 0;
                }
                pk.sequenceData[c] = dat;
            }
            return pk;
        }

        ~InterfaceBCISerial()
        {
            if(comPort != null)
            {
                comPort.Close();
            }
        }

        public void SetChannelProperties(BCIChannel ch)
        {
            if(comPort == null)
            {
                throw new ApplicationException("SetChannelProperties: Serial port is not open");
            }
            
            StringBuilder sb = new StringBuilder(":C");//channel set command
            sb.Append((char)('0' + ch.number));
            switch (ch.mode)
            {
                case BCIChannel.Mode.off:
                    sb.Append('0');
                    break;
                case BCIChannel.Mode.input:
                    sb.Append('i');
                    break;
                case BCIChannel.Mode.shorted:
                    sb.Append('s');
                    break;
                case BCIChannel.Mode.temp:
                    sb.Append('t');
                    break;
                case BCIChannel.Mode.test:
                    sb.Append('c');
                    break;
                case BCIChannel.Mode.mvdd:
                    sb.Append('m');
                    break;
                case BCIChannel.Mode.bias_meas:
                    sb.Append('b');
                    break;
                case BCIChannel.Mode.bias_n:
                    sb.Append('n');
                    break;
                case BCIChannel.Mode.bias_p:
                    sb.Append('p');
                    break;
            }
            if (ch.SRB2 == true)
            {
                sb.Append('1');
            }
            else
            {
                sb.Append('0');
            }
            switch (ch.gain)
            {
                case BCIChannel.Gain.x1:
                    sb.Append('1');
                    break;
                case BCIChannel.Gain.x2:
                    sb.Append('2');
                    break;
                case BCIChannel.Gain.x4:
                    sb.Append('4');
                    break;
                case BCIChannel.Gain.x6:
                    sb.Append('6');
                    break;
                case BCIChannel.Gain.x12:
                    sb.Append('c');
                    break;
                case BCIChannel.Gain.x24:
                    sb.Append('o');
                    break;
            }

            comPort.WriteLine(sb.ToString());
        }

        public void SetDatarate()
        {
            char r = 'a';
            switch (myBoard.dataRate)
            {
                case BCIBoard.DataRate.x250:
                    r = 'a';
                    break;
                case BCIBoard.DataRate.x500:
                    r = 'b';
                    break;
                case BCIBoard.DataRate.x1000:
                    r = 'c';
                    break;
                case BCIBoard.DataRate.x2000:
                    r = 'd';
                    break;
            }

            comPort.WriteLine(":R" + r);
        }

        public void SetProps()
        {
            char s = '0', l = '0', m = '0';
            if (myBoard.SRB1)
                s = '1';
            if (myBoard.LeadOffComp)
                l = '1';
            if (myBoard.BiasMeas)
                m = '1';

            comPort.WriteLine(":Pa" + m + s + l);
        }

        public void SetBiasgen()
        {
            comPort.WriteLine(":B" + makeBiasString());
        }

        public void SetChSequence()
        {
            StringBuilder sb = new StringBuilder(":Q");
            for (int i = 0; i < 16; i++)
            {
                char c = (char)(myBoard.channelSequence[i] + '0');
                sb.Append(c);
            }

            comPort.WriteLine(sb.ToString());
        }

        public void SetLeadOffChannels()
        {
            comPort.WriteLine(":O" + makeLOString(myBoard.leadOffChannel1) + makeLOString(myBoard.leadOffChannel2));
        }

        public void SetLeadOffSignal()
        {
            char f = '0', a = '0';
            switch (myBoard.leadOffSignalFreq)
            {
                case BCIBoard.LeadOffSignalFreq.DC:
                    f = '0';
                    break;
                case BCIBoard.LeadOffSignalFreq.AC7_8:
                    f = '1';
                    break;
                case BCIBoard.LeadOffSignalFreq.AC31_2:
                    f = '2';
                    break;
                case BCIBoard.LeadOffSignalFreq.DATARATE_DIV_4:
                    f = '3';
                    break;
            }
            switch (myBoard.leadOffSignalAmpl)
            {
                case BCIBoard.LeadOffSignalAmpl.nA6:
                    a = '0';
                    break;
                case BCIBoard.LeadOffSignalAmpl.nA24:
                    a = '1';
                    break;
                case BCIBoard.LeadOffSignalAmpl.uA6:
                    a = '2';
                    break;
                case BCIBoard.LeadOffSignalAmpl.uA24:
                    a = '3';
                    break;
            }

            comPort.WriteLine(":La" + f + a + '0');
        }

        public void SetTestSignal()
        {
            char f = '0', a = '0';
            switch (myBoard.testSignalFreq)
            {
                case BCIBoard.TestSignalFreq.DC:
                    f = '0';
                    break;
                case BCIBoard.TestSignalFreq.AC0_975:
                    f = '1';
                    break;
                case BCIBoard.TestSignalFreq.AC1_95:
                    f = '2';
                    break;
            }
            switch (myBoard.testSignalAmpl)
            {
                case BCIBoard.TestSignalAmpl.mV1_875:
                    a = '0';
                    break;
                case BCIBoard.TestSignalAmpl.mV3_75:
                    a = '1';
                    break;
            }

            comPort.WriteLine(":Ta" + f + a);
        }

        string makeLOString(BCIBoard.LeadOffChannel ch)
        {
            switch (ch)
            {
                case BCIBoard.LeadOffChannel.c1n:
                    return "1n";
                    break;
                case BCIBoard.LeadOffChannel.c2n:
                    return "2n";
                    break;
                case BCIBoard.LeadOffChannel.c3n:
                    return "3n";
                    break;
                case BCIBoard.LeadOffChannel.c4n:
                    return "4n";
                    break;
                case BCIBoard.LeadOffChannel.c5n:
                    return "5n";
                    break;
                case BCIBoard.LeadOffChannel.c6n:
                    return "6n";
                    break;
                case BCIBoard.LeadOffChannel.c7n:
                    return "7n";
                    break;
                case BCIBoard.LeadOffChannel.c8n:
                    return "8n";
                    break;
                case BCIBoard.LeadOffChannel.c1p:
                    return "1p";
                    break;
                case BCIBoard.LeadOffChannel.c2p:
                    return "2p";
                    break;
                case BCIBoard.LeadOffChannel.c3p:
                    return "3p";
                    break;
                case BCIBoard.LeadOffChannel.c4p:
                    return "4p";
                    break;
                case BCIBoard.LeadOffChannel.c5p:
                    return "5p";
                    break;
                case BCIBoard.LeadOffChannel.c6p:
                    return "6p";
                    break;
                case BCIBoard.LeadOffChannel.c7p:
                    return "7p";
                    break;
                case BCIBoard.LeadOffChannel.c8p:
                    return "8p";
                    break;
            }
            return "00";
        }

        string makeBiasString()
        {
            StringBuilder s = new StringBuilder(16);
            if (myBoard.Bias1n)
            {
                if (myBoard.Bias1p)
                    s.Append('1');
                else
                    s.Append('n');
            }
            else
            {
                if (myBoard.Bias1p)
                    s.Append('p');
                else
                    s.Append('0');
            }
            if (myBoard.Bias2n)
            {
                if (myBoard.Bias2p)
                    s.Append('1');
                else
                    s.Append('n');
            }
            else
            {
                if (myBoard.Bias2p)
                    s.Append('p');
                else
                    s.Append('0');
            }
            if (myBoard.Bias3n)
            {
                if (myBoard.Bias3p)
                    s.Append('1');
                else
                    s.Append('n');
            }
            else
            {
                if (myBoard.Bias3p)
                    s.Append('p');
                else
                    s.Append('0');
            }
            if (myBoard.Bias4n)
            {
                if (myBoard.Bias4p)
                    s.Append('1');
                else
                    s.Append('n');
            }
            else
            {
                if (myBoard.Bias4p)
                    s.Append('p');
                else
                    s.Append('0');
            }
            if (myBoard.Bias5n)
            {
                if (myBoard.Bias5p)
                    s.Append('1');
                else
                    s.Append('n');
            }
            else
            {
                if (myBoard.Bias5p)
                    s.Append('p');
                else
                    s.Append('0');
            }
            if (myBoard.Bias6n)
            {
                if (myBoard.Bias6p)
                    s.Append('1');
                else
                    s.Append('n');
            }
            else
            {
                if (myBoard.Bias6p)
                    s.Append('p');
                else
                    s.Append('0');
            }
            if (myBoard.Bias7n)
            {
                if (myBoard.Bias7p)
                    s.Append('1');
                else
                    s.Append('n');
            }
            else
            {
                if (myBoard.Bias7p)
                    s.Append('p');
                else
                    s.Append('0');
            }
            if (myBoard.Bias8n)
            {
                if (myBoard.Bias8p)
                    s.Append('1');
                else
                    s.Append('n');
            }
            else
            {
                if (myBoard.Bias8p)
                    s.Append('p');
                else
                    s.Append('0');
            }
            s.Append("00000000");
            return s.ToString();
        }

        public void StartData()
        {
            comPort.WriteLine(":S");
            isStreaming = true;
        }

        public void StopData()
        {
            comPort.WriteLine(":F");
            isStreaming = false;
        }
    }
}
