﻿using SciChart.Charting.Model.DataSeries;
using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Windows;
using System.Windows.Controls;
using System.Windows.Data;
using System.Windows.Documents;
using System.Windows.Input;
using System.Windows.Media;
using System.Windows.Media.Imaging;
using System.Windows.Shapes;

namespace ECGDisplay
{
    /// <summary>
    /// Interaction logic for ControlPlotter.xaml
    /// </summary>
    /// 
    public partial class ControlPlotter : Window, IBCIListener
    {
        public ObservableCollection<BCIPlot> BCIPlots //plots to be displayed
        {
            get { return _bci_plots; }
        }

        public bool PlotterEnabled { get; set; }

        ObservableCollection<BCIPlot> _bci_plots;

        public ControlPlotter()
        {
            InitializeComponent();

            PlotterEnabled = false;

            this.DataContext = this;

            BCIChannel.Number n1 = BCIChannel.Number.c5; //this one is left arm to right arm
            BCIChannel.Number n2 = BCIChannel.Number.c6; //this one is left leg to right arm

            _bci_plots = new ObservableCollection<BCIPlot>();
            /*_bci_plots.Add(new ECGPlot(ECGPlot.ECGLeadType.I,   n1, n2));
            _bci_plots.Add(new ECGPlot(ECGPlot.ECGLeadType.II,  n1, n2));
            _bci_plots.Add(new ECGPlot(ECGPlot.ECGLeadType.III, n1, n2));
            _bci_plots.Add(new ECGPlot(ECGPlot.ECGLeadType.aVR, n1, n2));
            _bci_plots.Add(new ECGPlot(ECGPlot.ECGLeadType.aVL, n1, n2));
            _bci_plots.Add(new ECGPlot(ECGPlot.ECGLeadType.aVF, n1, n2));*/
            _bci_plots.Add(new OneChPlot(BCIChannel.Number.c1));
            _bci_plots.Add(new OneChPlot(BCIChannel.Number.c2));
        }
        
        /*private void Rnd_Button_Click(object sender, RoutedEventArgs e)
        {
            SciChart.Core.Helpers.FasterRandom fr = new SciChart.Core.Helpers.FasterRandom();
            for (int i = 0; i < 110; i++)
            {
                double v = fr.NextDouble();
                _eeg_plots[0].AddPoint(plot_time, v);
                plot_time += (1.0 / 250);
            }
        }*/

        public void Initialize()
        {
            this.Show();
        }

        public void Shutdown()
        {
            this.Close();
        }

        public void AnalyzePacket(BCIPacket pk)
        {
            if (!PlotterEnabled)
                return;

            foreach (BCIPlot ep in _bci_plots)
            {
                ep.PullDataFrom(pk);
            }
        }
    }

    public interface BCIPlot
    {
        IXyDataSeries<double, double> OldData { get; set; }
        IXyDataSeries<double, double> NewData { get; set; }

        void PullDataFrom(BCIPacket pk);
        void AddPoint(double t, double v);
    }

    public class OneChPlot : BCIPlot
    {
        public IXyDataSeries<double, double> OldData { get; set; }
        public IXyDataSeries<double, double> NewData { get; set; }

        public BCIChannel.Number PlotChannel { get; }
        double lastChartTime = 0, lastPointTime = 0; //for adding incoming data at proper visual location

        public OneChPlot(BCIChannel.Number chan)
        {
            OldData = new XyDataSeries<double, double>(1000);
            NewData = new XyDataSeries<double, double>(1000);
            PlotChannel = chan;
        }
        
        public void PullDataFrom(BCIPacket pk)
        {
            foreach (BCIDataPoint bd in pk.sequenceData)
            {
                if (bd.channelNumber == PlotChannel)
                {
                    AddPoint(bd.time, bd.value);
                }
            }
        }

        public void AddPoint(double t, double v)
        {
            if(t - lastChartTime > 4.0 || t < lastChartTime)
            {
                lastChartTime = Math.Floor(t / 4.0) * 4.0;
                //shift any new data into the background
                OldData.Clear();
                OldData.Append(NewData.XValues, NewData.YValues);
                NewData.Clear();
                lastPointTime = 0;
            }

            t = t - lastChartTime;
            if(t < lastPointTime)
            {
                NewData.Clear();
            }

            NewData.Append(t, v);
            lastPointTime = t;
        }
    }

    public class ECGPlot : BCIPlot
    {
        public IXyDataSeries<double, double> OldData { get; set; }
        public IXyDataSeries<double, double> NewData { get; set; }

        public enum ECGLeadType { I, II, III, aVR, aVL, aVF };
        
        public ECGLeadType ECGLead { get; }
        double lastChartTime = 0, lastPointTime = 0; //for adding incoming data at proper visual location
        BCIChannel.Number _LARA, _LLRA, _LL;

        public ECGPlot(ECGLeadType lead, BCIChannel.Number LARA, BCIChannel.Number LLRA)
        {
            OldData = new XyDataSeries<double, double>(1000);
            NewData = new XyDataSeries<double, double>(1000);
            ECGLead = lead;
            _LARA = LARA; //voltage from left arm (+) to right arm (-), ECG standard lead I
            _LLRA = LLRA; //voltage from left leg (+) to right arm (-), ECG standard lead II
        }

        public void PullDataFrom(BCIPacket pk)
        {
            double leadI = 0, leadII = 0, t = 0;
            int leads = 0;
            foreach (BCIDataPoint bd in pk.sequenceData)
            {
                if(leads == 3)
                {
                    break;
                }
                if (bd.channelNumber == _LARA)
                {
                    leadI = bd.value;
                    t = bd.time;
                    leads |= 0x01;
                    continue;
                }
                if (bd.channelNumber == _LLRA)
                {
                    leadII = bd.value;
                    leads |= 0x02;
                }
            }
            if(leads == 3)
            {
                switch (ECGLead)
                {
                    case ECGLeadType.I:
                        AddPoint(t, leadI);
                        break;
                    case ECGLeadType.II:
                        AddPoint(t, leadII);
                        break;
                    case ECGLeadType.III:
                        AddPoint(t, leadII - leadI);
                        break;
                    case ECGLeadType.aVR:
                        AddPoint(t, (leadI + leadII) * -0.5);
                        break;
                    case ECGLeadType.aVL:
                        AddPoint(t, leadI - leadII * 0.5);
                        break;
                    case ECGLeadType.aVF:
                        AddPoint(t, leadII - leadI * 0.5);
                        break;
                }
            }
        }

        public void AddPoint(double t, double v)
        {
            if (t - lastChartTime > 4.0 || t < lastChartTime)
            {
                lastChartTime = Math.Floor(t / 4.0) * 4.0;
                //shift any new data into the background
                OldData.Clear();
                OldData.Append(NewData.XValues, NewData.YValues);
                NewData.Clear();
                lastPointTime = 0;
            }

            t = t - lastChartTime;
            if (t < lastPointTime)
            {
                NewData.Clear();
            }

            NewData.Append(t, v);
            lastPointTime = t;
        }
    }

    public class AuxPlot : BCIPlot
    {
        public IXyDataSeries<double, double> OldData { get; set; }
        public IXyDataSeries<double, double> NewData { get; set; }

        public enum AuxType { AccelX, AccelY, AccelZ };

        public AuxType PlotType { get; }
        double lastChartTime = 0, lastPointTime = 0; //for adding incoming data at proper visual location

        public AuxPlot(AuxType type)
        {
            OldData = new XyDataSeries<double, double>(1000);
            NewData = new XyDataSeries<double, double>(1000);
            PlotType = type;
        }

        public void PullDataFrom(BCIPacket pk)
        {
            switch (PlotType)
            {
                case AuxType.AccelX:
                    AddPoint(pk.packetTime, pk.accelX);
                    break;
                case AuxType.AccelY:
                    AddPoint(pk.packetTime, pk.accelY);
                    break;
                case AuxType.AccelZ:
                    AddPoint(pk.packetTime, pk.accelZ);
                    break;
            }
        }

        public void AddPoint(double t, double v)
        {
            if (t - lastChartTime > 4.0 || t < lastChartTime)
            {
                lastChartTime = Math.Floor(t / 4.0) * 4.0;
                //shift any new data into the background
                OldData.Clear();
                OldData.Append(NewData.XValues, NewData.YValues);
                NewData.Clear();
                lastPointTime = 0;
            }

            t = t - lastChartTime;
            if (t < lastPointTime)
            {
                NewData.Clear();
            }

            NewData.Append(t, v);
            lastPointTime = t;
        }
    }
}
