﻿using SciChart.Charting.Model.DataSeries;
using System;
using System.Windows;
using FFTWSharp;
using System.Runtime.InteropServices;

namespace ECGDisplay
{
    /// <summary>
    /// Interaction logic for ControlFFT.xaml
    /// </summary>
    public partial class ControlFFT : Window, IBCIListener
    {
        public IXyDataSeries<double, double> OldFFT { get; set; }
        public IXyDataSeries<double, double> NewFFT { get; set; }

        public bool FFTEnabled { get; set; }

        double[] data_in; //data to pass to FFT
        double[] fftd_data; //data after FFT has been called, complex format [re][im]
        double[] fftd_amplitudes; //amplitude only of FFT data
        double[] fftd_frequencies; //frequencies corresponding to amplitudes
        int data_in_ctr = 0;

        GCHandle fft_in, fft_out; //when reserving memory for FFT, ensure it will be unreserved to avoid memory eating
        IntPtr fftPlan;
        bool destroyHandles = false;

        BCIChannel.Number myChannel = BCIChannel.Number.c1;

        enum MissingDataMode { Average, Copy, Drop }; //what to do if received data is not uniform in time
        MissingDataMode missingMode = MissingDataMode.Average;
        int missed_times_ctr = 0;
        double dt = 1.0 / 250;
        double lastTime = -10;
        double lastValue = 0;

        public ControlFFT(int seconds, BCIBoard.DataRate dataRate, BCIChannel.Number chan)
        {
            InitializeComponent();

            if(seconds > 1000 || seconds < 1)
            {
                throw new ArgumentOutOfRangeException("ControlFFT: seconds");
            }

            myChannel = chan;

            //FFT wrapper and example: Github tszalay/FFTWSharp
            //Importing wisdom (wisdom speeds up the plan creation process, if that plan was previously created at least once)
            //fftwf.import_wisdom_from_filename("wisdom.wsd");

            int numPts = seconds * (int)dataRate;
            dt = 1.0 / (int)dataRate;

            data_in = new double[numPts];
            fftd_data = new double[numPts + 2];
            int pl = (numPts + 2) / 2;
            fftd_frequencies = new double[pl];

            //max frequency is dataRate / 2
            double df = 1.0 / seconds;

            for (int f = 0; f < pl; f++)
            {
                fftd_frequencies[f] = f * df;
            }
            fftd_amplitudes = new double[pl];

            fft_in = GCHandle.Alloc(data_in, GCHandleType.Pinned); //must be freed later
            fft_out = GCHandle.Alloc(fftd_data, GCHandleType.Pinned); //must be freed later
            fftPlan = fftw.dft_r2c_1d(numPts, fft_in.AddrOfPinnedObject(), fft_out.AddrOfPinnedObject(), fftw_flags.Estimate);
            destroyHandles = true;

            OldFFT = new XyDataSeries<double, double>(pl);
            NewFFT = new XyDataSeries<double, double>(pl);

            FFTEnabled = false;

            this.DataContext = this;            
        }

        ~ControlFFT()
        {
            if (destroyHandles)
            {
                //fftw.export_wisdom_to_filename("wisdom.wsd");
                fftw.destroy_plan(fftPlan);
                fft_in.Free();
                fft_out.Free();
            }
        }

        public void Initialize()
        {
            this.Show();
        }

        public void Shutdown()
        {
            if (destroyHandles)
            {
                //fftw.export_wisdom_to_filename("wisdom.wsd");
                fftw.destroy_plan(fftPlan);
                fft_in.Free();
                fft_out.Free();
                destroyHandles = false;
            }

            this.Close();
        }

        public void ResetData()
        {
            data_in_ctr = 0;
            lastTime = -10;
        }

        public void AnalyzePacket(BCIPacket pk)
        {
            if (!FFTEnabled)
                return;

            foreach (BCIDataPoint bd in pk.sequenceData)
            {
                if (bd.channelNumber == myChannel)
                {
                    if (lastTime > 0)
                    {
                        if (bd.time - lastTime > dt * 1.5)
                        {
                            missed_times_ctr++;
                            this.Dispatcher.Invoke(() => { MissedPtsText.Text = missed_times_ctr.ToString(); });
                            if (bd.time - lastTime > 0.064)
                            {
                                throw new Exception("Too large time gap, cannot analyze: " + bd.time.ToString() + ", " + lastTime.ToString());
                            }
                            //need to fill in missing data
                            switch (missingMode)
                            {
                                case MissingDataMode.Average:
                                    double s = (bd.value - lastValue) / (bd.time - lastTime); //slope
                                    while (bd.time - lastTime > dt * 1.5)
                                    {
                                        lastValue = lastValue + s * dt;
                                        lastTime = lastTime + dt;
                                        AddPoint(lastValue);
                                    }
                                    break;
                                case MissingDataMode.Copy:
                                    while (bd.time - lastTime > dt * 1.5)
                                    {
                                        lastTime = lastTime + dt;
                                        AddPoint(lastValue);
                                    }
                                    break;
                                case MissingDataMode.Drop:
                                    //don't do anything
                                    break;
                            }
                        }
                    }
                    AddPoint(bd.value);
                    lastValue = bd.value;
                    lastTime = bd.time;
                }
            }
        }

        void AddPoint(double val)
        {
            data_in[data_in_ctr] = val;
            data_in_ctr++;
            if(data_in_ctr >= data_in.Length)
            {
                AnalyzeFFT();
                data_in_ctr = 0;
            }
            if(data_in_ctr % 100 == 0)
            {
                this.Dispatcher.Invoke(() => { LoadedPtsText.Text = data_in_ctr.ToString(); });
            }
        }

        void AnalyzeFFT()
        {
            fftw.execute(fftPlan); //do FFT transform from data_in to fftd_data
            for (int f = 0; f < fftd_amplitudes.Length; f++)
            {
                fftd_amplitudes[f] = Math.Sqrt(fftd_data[f * 2] * fftd_data[f * 2] + fftd_data[f * 2 + 1] * fftd_data[f * 2 + 1]) * 2.0 / (fftd_data.Length - 2); //amplitude of FFT (element 0 and n need to be not multiplied by 2 for accurate amplitudes due to folding of positive and negative spectrum here)
            }
            fftd_amplitudes[0] *= 0.5;
            fftd_amplitudes[fftd_amplitudes.Length - 1] *= 0.5;

            if (NewFFT.XValues != null)
            {
                //shift any existing data into the background
                OldFFT.Clear();
                OldFFT.Append(NewFFT.XValues, NewFFT.YValues);
                NewFFT.Clear();
            }

            NewFFT.Append(fftd_frequencies, fftd_amplitudes);
        }
    }
}
