﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Windows;
using System.Windows.Controls;
using System.Windows.Media;
using System.Xml.Serialization;

namespace PhotoArrange
{
    public class CBoard
    {
        public string name;
        public Brush background;
        public List<CBImage> nodes;
        public List<CBLine> vertices;

        public enum MouseMode { None, DragImage, NewLineEnd, NewImage };
        public enum Tools { None, Select, Drag, Line };

        [XmlIgnore] public MouseMode dragMode = MouseMode.None;
        [XmlIgnore] public Tools clickMode = Tools.None;

        private CBLine newLine;
        [XmlIgnore] public object selectedObject;
        [XmlIgnore] public string filepath;

        public double Width = 1000, Height = 1000;
        //[XmlIgnore] public double Zoom = DefaultZoomAbs;
        [XmlIgnore] public bool LineFollowsImg1 = false;
        [XmlIgnore] public bool LineCarriesImg1 = false;
        [XmlIgnore] public bool LineDuplicatesAllowed = false;
        public static readonly Color DefaultColorAbs = Colors.Black;
        public static readonly double DefaultThicknessAbs = 2;
        public static readonly double DefaultZoomAbs = 1;
        /*[XmlIgnore] public double DefaultThickness = 2;
        [XmlIgnore] public string DefaultColor = "#FF000000";*/

        private Canvas myCanvas;

        public CBoard()
        {
            nodes = new List<CBImage>();
            vertices = new List<CBLine>();
            CBImage.ResetDefaults();
            CBLine.ResetDefaults();
        }
        
        public void SetCanvas(Canvas canv)
        {
            myCanvas = canv;
        }

        public void LoadFromXML(Canvas canv, string mypath)
        {
            myCanvas = canv;
            filepath = mypath;
            canv.Width = Width;
            canv.Height = Height;
            foreach (CBImage cbi in nodes)
            {
                cbi.LoadFromXML(canv, this, System.IO.Path.GetDirectoryName(filepath));
            }
            foreach(CBLine cbl in vertices)
            {
                cbl.LoadFromXML(canv, this);
            }
        }

        public void PrepareSaveXML(string mypath)
        {
            filepath = mypath;
            Width = myCanvas.Width;
            Height = myCanvas.Height;
            foreach (CBImage cbi in nodes)
            {
                cbi.ReferenceFilename(System.IO.Path.GetDirectoryName(filepath));
            }
        }

        public void StartNewLine(CBImage img1, Point pos) //gets called by image clicked on
        {
            dragMode = MouseMode.NewLineEnd;
            Select(img1);
            newLine = new CBLine();
            newLine.StartNewLine(myCanvas, this, img1, pos);
            dragMode = MouseMode.NewLineEnd; //for reference in main canvas mouse move and others
        }

        public void PreviewNewLine(Point pos) //gets called by main canvas mouse move
        {
            newLine.MoveEnd(false, pos.X, pos.Y);
        }

        public void PreviewDrag(Point pos) //gets called by main canvas mouse move
        {
            if (selectedObject is CBImage)
                ((CBImage)selectedObject).MoveOutline(pos);
        }

        public void FinishDrag(Point pos) //gets called by main canvas mouse up
        {
            if (selectedObject is CBImage)
            {
                CBImage cbi = ((CBImage)selectedObject);
                cbi.MoveOutline(pos);
                cbi.MoveImage(pos);
                cbi.RelocateLines();
                Deselect();
                dragMode = MouseMode.None;
            }
        }

        public void CancelNewLine() //gets called by main canvas mouse up
        {
            newLine.ShowLine(false);
            Deselect();
            dragMode = MouseMode.None;
            newLine = null;
        }

        public void FinishNewLine(CBImage img2) //gets called by image mouse up
        {
            if (!LineDuplicatesAllowed && newLine.CheckDuplicateNew(img2))
            { //this is a duplicate line so ignore
                CancelNewLine();
                System.Windows.MessageBox.Show("Error: Line between these nodes already exists.");
                return;
            }
            newLine.FinishNewLine(img2);
            Deselect();
            dragMode = MouseMode.None;
            vertices.Add(newLine);
            newLine = null;
        }

        public void Select(CBLine line)
        {
            Deselect();
            line.MarkSelected(true);
            selectedObject = line;
        }

        public void Select(CBImage img)
        {
            Deselect();
            img.MarkSelected(true);
            selectedObject = img;
        }

        public void Deselect()
        {
            if (selectedObject == null)
                return;
            if(selectedObject is CBLine)
                ((CBLine)selectedObject).MarkSelected(false);
            if (selectedObject is CBImage)
                ((CBImage)selectedObject).MarkSelected(false);
            selectedObject = null;
        }

        public void DeleteSelected()
        {
            if (selectedObject == null)
                return;
            if (selectedObject is CBLine)
            {
                ((CBLine)selectedObject).Delete();
                vertices.Remove((CBLine)selectedObject);
            }
            if (selectedObject is CBImage)
            {
                ((CBImage)selectedObject).Delete();
                nodes.Remove((CBImage)selectedObject);
            }
            selectedObject = null;
        }

        public void Unload()
        {
            foreach (CBImage cbi in nodes)
            {
                cbi.Delete();
            }
            vertices.Clear();
            nodes.Clear();
        }
    }
}
