//Teensy LC, Serial

#include <EEPROM.h>

#include "OctoWS2811_reduced.h"
#include <Wire.h>

#include "spherical_coeffs.h"

const int ledsPerStrip = 49;
const int totalLEDs = ledsPerStrip*4;
DMAMEM uint32_t displayMemory[ledsPerStrip*6];
uint32_t drawingMemory[ledsPerStrip*6];
OctoWS2811 leds(ledsPerStrip, displayMemory, drawingMemory, WS2811_GRB | WS2811_800kHz);

elapsedMillis startup_millis;

const int LEDPowerPin=24, SSegEnablePin=25, PowerGoodPin=3, PowerAlertPin=4;

int errorState=0, visualizeMode=0, brightMode=0, menu7=0, submenu7=0;

//driving 7-segment 4 digits LED display
//each segment is connected through a resistor to a driver capable of outputting the necessary current
//each digit lights up at once
//int spins[]={17,16,15,13,12,11,10,9}; //pins for 7 segments + dot of each digit on 20231205 board
int spins[]={9,10,11,12,13,15,16,17}; //pins for 7 segments + dot of each digit on 20240114 board
int dpins[]={5,6,20,21}; //pins for each digit on 20231205 and 20240114 boards, 20240114 is inverted

int cdigit=0; //currently displayed digit
byte sflags[]={0,0,0,0}; //flags for segments on each digit
//    1
// 32    2
//    64
// 16    4
//    8   128
const byte symv=4|8|16; // v symbol
const byte symA=1|2|4|16|32|64; // A symbol
const byte symC=1|8|16|32; // C
const byte symt=8|16|32|64; //t
const byte symE=1|8|16|32|64; //E
const byte symu=2|4|8|16|32; //u
const byte symb=4|8|16|32|64; //b
const byte symd=2|4|8|16|64;
const byte symr=16|64; //r
const byte symL=8|16|32;
const byte symF=1|16|32|64;
const byte symP=1|2|16|32|64;
const byte symH=2|4|16|32|64;
const byte sdot=128;
const byte sdash=64;
const byte iflags[]={1|2|4|8|16|32, 2|4, 1|2|64|16|8, 1|2|64|4|8, 32|64|2|4, 1|32|64|4|8, 1|32|64|4|8|16, 1|2|4, 1|2|4|8|16|32|64, 1|2|4|8|32|64, symA, symb, symC, symd, symE, symF}; //0 to 9, A to F

uint32_t mmc_in[3], mmc_avarr[3];
int32_t mmc_svarr[3];
float mmc_boff[3]={-8164.84f,-8210.20f,-8250.42f}, mmc_soff[3]={-48.560f,-45.263f,14.514f}, mmc_out[4];
//later calibrated: Existing bridge offsets [mG] X,Y,Z -8155.19,-8234.94,-8241.57  Existing zero offsets [mG] X,Y,Z -48.91,-45.35,14.35

const byte MMC_addr=0b00110000;
byte mmc_buf[7];
elapsedMicros mmc_conversion_start;
const uint32_t mmc_delayms=11, mmc_delaymsfast=11;
int mmc_conversion_active=0;
const int mmc_niceavg=8, mmc_intgavg=10;
float cos_stamp, sin_stamp, mmc_intgarr[6];

//touch button processor
const int numTouchBtns=2;
uint_fast8_t touchBtnPin[numTouchBtns]={1,0}; //as labeled on teensy LC board
uint_fast8_t touchBtnTSI[numTouchBtns]={10,9}; //different from labeled pins above by pin2tsi[] from touch.c for __MKL26Z64__
uint_fast16_t touchBtnThres=140;
uint_fast16_t touchBtnHyst=20;
int touchBtnEvent[numTouchBtns]={0,0};
bool touchBtnPressed[numTouchBtns]={0,0};
uint_fast16_t touchBtnValue[numTouchBtns]={0,0};
int touchBtnAcqState=0, touchBtnI=0;
elapsedMicros touchBtnAcqTime;
const int touchBtnDelay=2000; //microseconds to wait between measurements


//LED spherical direction output code
//LED strips are 49 LEDs long in memory; for strips other than the first, the 49th LED is not connected (all meet at Z+)
//the first strip of LEDs goes from Z- through X+ to Z+ including Z+, second goes through Y+, third goes through X-, fourth through Y-

//the mapping from MMC sensor to sphere is as follows:
//sensor Z+ = sphere X-
//sensor Y+ = sphere Z+
//sensor X+ = sphere Y+
//the coordinates below are in sphere basis

const int antiLEDs=97;//it is possible to cover the entire sphere by calculating values for LEDs 0 to 96 and applying antipodal symmetry
const float xleds[totalLEDs]={0.167277,0.382683,0.535467,0.707107,0.535467,0.325058,0.464385,0.57735,0.661225,0.827549,0.827549,0.661225,0.754117,0.888074,0.827549,0.92388,0.827549,0.888074,0.754117,0.661225,0.827549,0.92388,0.971616,1.0,0.971616,0.92388,0.971616,0.888074,0.754117,0.661225,0.827549,0.707107,0.827549,0.92388,0.971616,0.888074,0.827549,0.661225,0.754117,0.661225,0.57735,0.464385,0.325058,0.535467,0.707107,0.535467,0.382683,0.167277,0.0,0.167277,0.0,-0.168634,0.0,0.168634,0.325058,0.464385,0.57735,0.354349,0.168634,-0.168634,-0.354349,-0.464385,-0.325058,-0.535467,-0.382683,-0.535467,-0.325058,-0.464385,-0.354349,-0.168634,0.0,-0.167277,0.0,-0.167277,0.0,0.167277,0.325058,0.464385,0.661225,0.535467,0.707107,0.535467,0.382683,0.167277,0.325058,0.168634,0.354349,0.464385,0.661225,0.57735,0.464385,0.325058,0.168634,0.0,-0.168634,0.0,0.167277, NAN ,-0.167277,-0.382683,-0.535467,-0.707107,-0.535467,-0.325058,-0.464385,-0.57735,-0.661225,-0.827549,-0.827549,-0.661225,-0.754117,-0.888074,-0.827549,-0.92388,-0.827549,-0.888074,-0.754117,-0.661225,-0.827549,-0.92388,-0.971616,-1.0,-0.971616,-0.92388,-0.971616,-0.888074,-0.754117,-0.661225,-0.827549,-0.707107,-0.827549,-0.92388,-0.971616,-0.888074,-0.827549,-0.661225,-0.754117,-0.661225,-0.57735,-0.464385,-0.325058,-0.535467,-0.707107,-0.535467,-0.382683,-0.167277, NAN ,-0.167277,0.0,0.168634,0.0,-0.168634,-0.325058,-0.464385,-0.57735,-0.354349,-0.168634,0.168634,0.354349,0.464385,0.325058,0.535467,0.382683,0.535467,0.325058,0.464385,0.354349,0.168634,0.0,0.167277,0.0,0.167277,0.0,-0.167277,-0.325058,-0.464385,-0.661225,-0.535467,-0.707107,-0.535467,-0.382683,-0.167277,-0.325058,-0.168634,-0.354349,-0.464385,-0.661225,-0.57735,-0.464385,-0.325058,-0.168634,0.0,0.168634,0.0,-0.167277, NAN };
//const float xleds2[antiLEDs]={0.0279816,0.1464463,0.2867249,0.5000003,0.2867249,0.1056627,0.2156534,0.333333,0.4372185,0.6848373,0.6848373,0.4372185,0.5686924,0.7886754,0.6848373,0.8535543,0.6848373,0.7886754,0.5686924,0.4372185,0.6848373,0.8535543,0.9440377,1.0,0.9440377,0.8535543,0.9440377,0.7886754,0.5686924,0.4372185,0.6848373,0.5000003,0.6848373,0.8535543,0.9440377,0.7886754,0.6848373,0.4372185,0.5686924,0.4372185,0.333333,0.2156534,0.1056627,0.2867249,0.5000003,0.2867249,0.1464463,0.0279816,0.0,0.0279816,0.0,0.0284374,0.0,0.0284374,0.1056627,0.2156534,0.333333,0.1255632,0.0284374,0.0284374,0.1255632,0.2156534,0.1056627,0.2867249,0.1464463,0.2867249,0.1056627,0.2156534,0.1255632,0.0284374,0.0,0.0279816,0.0,0.0279816,0.0,0.0279816,0.1056627,0.2156534,0.4372185,0.2867249,0.5000003,0.2867249,0.1464463,0.0279816,0.1056627,0.0284374,0.1255632,0.2156534,0.4372185,0.333333,0.2156534,0.1056627,0.0284374,0.0,0.0284374,0.0,0.0279816}; //squared values for distance calculation
const float yleds[totalLEDs]={-0.167277,0.0,0.168634,0.0,-0.168634,-0.325058,-0.464385,-0.57735,-0.354349,-0.168634,0.168634,0.354349,0.464385,0.325058,0.535467,0.382683,0.535467,0.325058,0.464385,0.354349,0.168634,0.0,0.167277,0.0,0.167277,0.0,-0.167277,-0.325058,-0.464385,-0.661225,-0.535467,-0.707107,-0.535467,-0.382683,-0.167277,-0.325058,-0.168634,-0.354349,-0.464385,-0.661225,-0.57735,-0.464385,-0.325058,-0.168634,0.0,0.168634,0.0,-0.167277,0.0,0.167277,0.382683,0.535467,0.707107,0.535467,0.325058,0.464385,0.57735,0.661225,0.827549,0.827549,0.661225,0.754117,0.888074,0.827549,0.92388,0.827549,0.888074,0.754117,0.661225,0.827549,0.92388,0.971616,1.0,0.971616,0.92388,0.971616,0.888074,0.754117,0.661225,0.827549,0.707107,0.827549,0.92388,0.971616,0.888074,0.827549,0.661225,0.754117,0.661225,0.57735,0.464385,0.325058,0.535467,0.707107,0.535467,0.382683,0.167277, NAN ,0.167277,0.0,-0.168634,0.0,0.168634,0.325058,0.464385,0.57735,0.354349,0.168634,-0.168634,-0.354349,-0.464385,-0.325058,-0.535467,-0.382683,-0.535467,-0.325058,-0.464385,-0.354349,-0.168634,0.0,-0.167277,0.0,-0.167277,0.0,0.167277,0.325058,0.464385,0.661225,0.535467,0.707107,0.535467,0.382683,0.167277,0.325058,0.168634,0.354349,0.464385,0.661225,0.57735,0.464385,0.325058,0.168634,0.0,-0.168634,0.0,0.167277, NAN ,-0.167277,-0.382683,-0.535467,-0.707107,-0.535467,-0.325058,-0.464385,-0.57735,-0.661225,-0.827549,-0.827549,-0.661225,-0.754117,-0.888074,-0.827549,-0.92388,-0.827549,-0.888074,-0.754117,-0.661225,-0.827549,-0.92388,-0.971616,-1.0,-0.971616,-0.92388,-0.971616,-0.888074,-0.754117,-0.661225,-0.827549,-0.707107,-0.827549,-0.92388,-0.971616,-0.888074,-0.827549,-0.661225,-0.754117,-0.661225,-0.57735,-0.464385,-0.325058,-0.535467,-0.707107,-0.535467,-0.382683,-0.167277, NAN };
//const float yleds2[antiLEDs]={0.0279816,0.0,0.0284374,0.0,0.0284374,0.1056627,0.2156534,0.333333,0.1255632,0.0284374,0.0284374,0.1255632,0.2156534,0.1056627,0.2867249,0.1464463,0.2867249,0.1056627,0.2156534,0.1255632,0.0284374,0.0,0.0279816,0.0,0.0279816,0.0,0.0279816,0.1056627,0.2156534,0.4372185,0.2867249,0.5000003,0.2867249,0.1464463,0.0279816,0.1056627,0.0284374,0.1255632,0.2156534,0.4372185,0.333333,0.2156534,0.1056627,0.0284374,0.0,0.0284374,0.0,0.0279816,0.0,0.0279816,0.1464463,0.2867249,0.5000003,0.2867249,0.1056627,0.2156534,0.333333,0.4372185,0.6848373,0.6848373,0.4372185,0.5686924,0.7886754,0.6848373,0.8535543,0.6848373,0.7886754,0.5686924,0.4372185,0.6848373,0.8535543,0.9440377,1.0,0.9440377,0.8535543,0.9440377,0.7886754,0.5686924,0.4372185,0.6848373,0.5000003,0.6848373,0.8535543,0.9440377,0.7886754,0.6848373,0.4372185,0.5686924,0.4372185,0.333333,0.2156534,0.1056627,0.2867249,0.5000003,0.2867249,0.1464463,0.0279816};
const float zleds[totalLEDs]={-0.971616,-0.92388,-0.827549,-0.707107,-0.827549,-0.888074,-0.754117,-0.57735,-0.661225,-0.535467,-0.535467,-0.661225,-0.464385,-0.325058,-0.168634,0.0,0.168634,0.325058,0.464385,0.661225,0.535467,0.382683,0.167277,0.0,-0.167277,-0.382683,-0.167277,-0.325058,-0.464385,-0.354349,-0.168634,0.0,0.168634,0.0,0.167277,0.325058,0.535467,0.661225,0.464385,0.354349,0.57735,0.754117,0.888074,0.827549,0.707107,0.827549,0.92388,0.971616,1.0,-0.971616,-0.92388,-0.827549,-0.707107,-0.827549,-0.888074,-0.754117,-0.57735,-0.661225,-0.535467,-0.535467,-0.661225,-0.464385,-0.325058,-0.168634,0.0,0.168634,0.325058,0.464385,0.661225,0.535467,0.382683,0.167277,0.0,-0.167277,-0.382683,-0.167277,-0.325058,-0.464385,-0.354349,-0.168634,0.0,0.168634,0.0,0.167277,0.325058,0.535467,0.661225,0.464385,0.354349,0.57735,0.754117,0.888074,0.827549,0.707107,0.827549,0.92388,0.971616, NAN ,-0.971616,-0.92388,-0.827549,-0.707107,-0.827549,-0.888074,-0.754117,-0.57735,-0.661225,-0.535467,-0.535467,-0.661225,-0.464385,-0.325058,-0.168634,0.0,0.168634,0.325058,0.464385,0.661225,0.535467,0.382683,0.167277,0.0,-0.167277,-0.382683,-0.167277,-0.325058,-0.464385,-0.354349,-0.168634,0.0,0.168634,0.0,0.167277,0.325058,0.535467,0.661225,0.464385,0.354349,0.57735,0.754117,0.888074,0.827549,0.707107,0.827549,0.92388,0.971616, NAN ,-0.971616,-0.92388,-0.827549,-0.707107,-0.827549,-0.888074,-0.754117,-0.57735,-0.661225,-0.535467,-0.535467,-0.661225,-0.464385,-0.325058,-0.168634,0.0,0.168634,0.325058,0.464385,0.661225,0.535467,0.382683,0.167277,0.0,-0.167277,-0.382683,-0.167277,-0.325058,-0.464385,-0.354349,-0.168634,0.0,0.168634,0.0,0.167277,0.325058,0.535467,0.661225,0.464385,0.354349,0.57735,0.754117,0.888074,0.827549,0.707107,0.827549,0.92388,0.971616, NAN };
//const float zleds2[antiLEDs]={0.9440377,0.8535543,0.6848373,0.5000003,0.6848373,0.7886754,0.5686924,0.333333,0.4372185,0.2867249,0.2867249,0.4372185,0.2156534,0.1056627,0.0284374,0.0,0.0284374,0.1056627,0.2156534,0.4372185,0.2867249,0.1464463,0.0279816,0.0,0.0279816,0.1464463,0.0279816,0.1056627,0.2156534,0.1255632,0.0284374,0.0,0.0284374,0.0,0.0279816,0.1056627,0.2867249,0.4372185,0.2156534,0.1255632,0.333333,0.5686924,0.7886754,0.6848373,0.5000003,0.6848373,0.8535543,0.9440377,1.0,0.9440377,0.8535543,0.6848373,0.5000003,0.6848373,0.7886754,0.5686924,0.333333,0.4372185,0.2867249,0.2867249,0.4372185,0.2156534,0.1056627,0.0284374,0.0,0.0284374,0.1056627,0.2156534,0.4372185,0.2867249,0.1464463,0.0279816,0.0,0.0279816,0.1464463,0.0279816,0.1056627,0.2156534,0.1255632,0.0284374,0.0,0.0284374,0.0,0.0279816,0.1056627,0.2867249,0.4372185,0.2156534,0.1255632,0.333333,0.5686924,0.7886754,0.6848373,0.5000003,0.6848373,0.8535543,0.9440377};
const byte antipodes[totalLEDs]={145,144,143,142,141,140,139,138,135,134,118,117,116,115,114,113,112,111,110,109,108,123,122,121,120,119,132,133,136,137,130,129,128,131,124,125,107,106,126,127,105,104,103,102,101,100,99,98,255, 194,193,192,191,190,189,188,187,184,183,167,166,165,164,163,162,161,160,159,158,157,172,171,170,169,168,181,182,185,186,179,178,177,180,173,174,156,155,175,176,154,153,152,151,150,149,148,147,255, 47,46,45,44,43,42,41,40,37,36,20,19,18,17,16,15,14,13,12,11,10,25,24,23,22,21,34,35,38,39,32,31,30,33,26,27,9,8,28,29,7,6,5,4,3,2,1,0,255, 96,95,94,93,92,91,90,89,86,85,69,68,67,66,65,64,63,62,61,60,59,74,73,72,71,70,83,84,87,88,81,80,79,82,75,76,58,57,77,78,56,55,54,53,52,51,50,49,255}; //refers to LED index, 255 means no antipode


int npb = 0x0F;
const int maxorder=6;
int lastnpj[maxorder]={0,0,0,0,0,0};
float lastq[maxorder];
float d_alpha=16, d_beta=1.0;

void displayNearestPoint(float x, float y, float z, int brt, int order){
	//x,y,z assumed to be normalized and in sphere coordinates
	for(int p=0; p<order; p++){
		leds.setPixel(lastnpj[p], 0);
		if(antipodes[lastnpj[p]]!=255){
			leds.setPixel(antipodes[lastnpj[p]], 0);
		}
		lastnpj[p]=0;
		lastq[p]=10;
	}
	//normalize
	/*float q=sqrt(x*x+y*y+z*z);
	x/=q;
	y/=q;
	z/=q;*/
	//float q=10;
	float dx,dy,dz,d2x,d2y,d2z,dq;
	int bc;
	//lastnpj=0;
	for(int i=0; i<totalLEDs; i++){
		dx=xleds[i]-x;
		dy=yleds[i]-y;
		dz=zleds[i]-z;
		dq=dx*dx+dy*dy+dz*dz;//no need for sqrt()
		dx=0;
		bc=255;
		for(int p=0; p<order; p++){
			dy=lastq[p]-dq;
			if(dy>dx){
				dx=dy;
			bc=p;
			}
		}
   if(bc!=255){
    lastq[bc]=dq;
    lastnpj[bc]=i;
   }
		show7sDigitTimeLimit(); //call here to avoid long delays between digit display updates
	}
	if(order==1){
    /*if(lastq[0]<=1e-4f){
      bc=brt;
    }else{
      bc=0;
    }*/
		leds.setPixel(lastnpj[0], (brt << 16));// | bc);
		if(antipodes[lastnpj[0]]!=255){
			leds.setPixel(antipodes[lastnpj[0]], (brt << 8));// | bc);
		}
	}else{
		dq=0;
    bc=255;
		for(int p=0; p<order; p++){
      if(lastq[p]<=1e-4f){
        bc=p;
        break;
      }
			//lastq[p]=max(lastq[p],1e-6f);
      lastq[p]=1.0f/(lastq[p]);
      /*if(brt>0x50){
        lastq[p]*=lastq[p];
      }*/
			dq=dq+lastq[p];
		}
    show7sDigitTimeLimit(); //call here to avoid long delays between digit display updates
    /*int de=0;
    bool sw=true;
    int r=order-1;
    while(sw){
      sw=false;
      for(int p=r; p>de; p--){
        if(lastq[p]<lastq[p-1]){
          sw=(p!=r);
          dy=lastq[p-1];
          lastq[p-1]=lastq[p];
          lastq[p]=dy;
          bc=lastnpj[p-1];
          lastnpj[p-1]=lastnpj[p];
          lastnpj[p]=bc;
        }
      }
      de++;
    }*/
    //show7sDigitTimeLimit(); //call here to avoid long delays between digit display updates
    if(bc!=255){
      leds.setPixel(lastnpj[bc], (brt << 16) | brt);
      if(antipodes[lastnpj[bc]]!=255){
        leds.setPixel(antipodes[lastnpj[bc]], (brt << 8) | brt);
      }
    }else{
  		for(int p=0; p<order; p++){
  			dx=lastq[p]/dq;
  			bc=round(brt*dx);
        //bc=bc&0xFF;
        //brt=brt-bc;
  			leds.setPixel(lastnpj[p], bc << 16);
  			if(antipodes[lastnpj[p]]!=255){
  				leds.setPixel(antipodes[lastnpj[p]], bc << 8);
  			}
  		}
    }
	}
  leds.show();
  show7sDigitTimeLimit();
}

/*int scaleColor(int color1, float scale){ //this is too slow
  int cr1=(color1>>16)&0xFF, cg1=(color1>>8)&0xFF, cb1=(color1)&0xFF;
  return ((((int)(cr1*scale))<<16)|(((int)(cg1*scale))<<8)|((int)(cb1*scale))) & 0x00FFFFFF;
}*/

void displaySphereGradient(float x, float y, float z, float alpha){//, float beta){
	//alpha sets maximum LED output code (0 to 255), beta sets power of gradient (default 1)
	//x,y,z assumed to be normalized
	/*float q=sqrt(x*x+y*y+z*z);
	x/=q;
	y/=q;
	z/=q;*/
	float q=1;
	float col;
    int cr, crp, cra;
	for(int i=0; i<antiLEDs; i++){
		//dot product
		q=(xleds[i]*x) + (yleds[i]*y) + (zleds[i]*z);
		if(q >= 0){
			col=alpha*(q*q*q);//pow(q, beta);
			cr=round(col);
			crp=(cr & 0xFF)<<16;
			cra=(cr & 0xFF);
		}else{
			col=alpha*(q*q*-q);//*pow(-q, beta);
			cr=round(col);
			crp=(cr & 0xFF);
			cra=(cr & 0xFF)<<16;
		}
		leds.setPixel(i, crp);
		if(antipodes[i]!=255){
			leds.setPixel(antipodes[i], cra);
		}
		show7sDigitTimeLimit(); //call here to avoid long delays between digit display updates
	}
  leds.show();
  show7sDigitTimeLimit();
}

void displaySphereHalo(float x, float y, float z, float alpha){//, float beta){
  //alpha sets maximum LED output code (0 to 255), beta sets power of gradient (default 1)
  //x,y,z assumed to be normalized
  /*float q=sqrt(x*x+y*y+z*z);
  x/=q;
  y/=q;
  z/=q;*/
  float q=1;
  float col;
  int cr, crp;
  for(int i=0; i<antiLEDs; i++){
    //dot product
    q=(xleds[i]*x) + (yleds[i]*y) + (zleds[i]*z);
    if(q >= 0){
      col=alpha*((1-q)*(1-q)*(1-q));//pow(1-q, beta);
    }else{
      col=alpha*((q+1)*(q+1)*(q+1));//pow(q+1, beta);
    }
	cr=round(col);
	crp=(cr & 0xFF)<<8;
    leds.setPixel(i, crp);
	if(antipodes[i]!=255){
		leds.setPixel(antipodes[i], crp);
	}
    show7sDigitTimeLimit(); //call here to avoid long delays between digit display updates
  }
  leds.show();
  show7sDigitTimeLimit();
}

void displayGrowingSphere(float x, float y, float z, float g, bool pos, float alpha){
  //alpha sets maximum LED output code (0 to 255)
  //x,y,z assumed to be normalized, g in range 0 to 1
  /*float q=sqrt(x*x+y*y+z*z);
  x/=q;
  y/=q;
  z/=q;*/
  float q=1;
  float cm=1-2*g;//cos(-g) takes too much space
  int cr=alpha/2;
  cr=cr|(cr<<8); //GB
  if(pos){
    cr<<=8; //RG
  }
  for(int i=0; i<antiLEDs; i++){
    //dot product
    q=(xleds[i]*x) + (yleds[i]*y) + (zleds[i]*z);
    if(!pos){
      q=-q;
    }
    if(q >= cm){
      leds.setPixel(i, cr);
    }else{
      leds.setPixel(i, 0);
    }
	if(antipodes[i]!=255){
        q=-q;
	    if(q >= cm){
		  leds.setPixel(antipodes[i], cr);
		}else{
		  leds.setPixel(antipodes[i], 0);
		}
    }
    show7sDigitTimeLimit(); //call here to avoid long delays between digit display updates
  }
  leds.show();
  show7sDigitTimeLimit();
}

void displaySphHCoeff(int index, float alpha){//, float beta){
  index=constrain(index,0,numsphc-1);
  index=index*totalLEDs;
  float q=1;
  float col;
  int cr;
  for(int i=0; i<totalLEDs; i++){
    q=sphc[index+i];
	if(isnan(q)){
      continue;
    }    
    if(q >= 0){
      col=alpha*(q*q*q);//pow(q, beta);
      cr=col;
      leds.setPixel(i, (cr & 0xFF)<<8);
    }else{
      col=alpha*(q*q*-q);//pow(-q, beta);
      cr=col;
      leds.setPixel(i, (cr & 0xFF));
    }
    show7sDigitTimeLimit(); //call here to avoid long delays between digit display updates
  }
  leds.show();
  show7sDigitTimeLimit();
}

int debugLEDctr=0;
elapsedMillis debugLEDtime=0;

void debugLEDs() {
  if(debugLEDtime<100){
    return;
  }
  leds.setPixel(debugLEDctr,0);
  debugLEDctr=(debugLEDctr+1)%totalLEDs;
  leds.setPixel(debugLEDctr,0x0F0000);
  leds.show();
  debugLEDtime=0;
}

void setup() {
  set7setup(); // uses pins defined in 7seg.h for 7 segment display
  load_cal();
  Wire.begin(); // uses pins 18,19 for MMC
  Wire1.begin(); //uses pins 22,23 for INA232
  ina232_start();
  mmc_start();
  pinMode(LEDPowerPin, OUTPUT);
  digitalWrite(LEDPowerPin, LOW); //turn off LEDs for now
  pinMode(SSegEnablePin, OUTPUT);
  digitalWrite(SSegEnablePin, LOW); //turn off 7seg display 5V output for now
  pinMode(PowerGoodPin, INPUT_PULLUP);
  //attach interrupt to PowerGoodPin to turn off LEDs on undervoltage
  
  leds.begin(); // uses pins 2,14,7,8 for LED strips 0,1,2,3
  touchBtnSetup(); //uses pins ? for touch button sensing
  mag_startup();
}

void load_cal(){
  byte v[4];
  
  v[0] = EEPROM.read(0);
  v[1] = EEPROM.read(1);
  v[2] = EEPROM.read(2);
  v[3] = EEPROM.read(3);
  if(v[0]!=0x11 || v[1]!=0x22 || v[2]!=0xA5){ //unique code to specify programming has occurred
    return;
  }
  if((v[3]&0x01)!=0){ //load bridge calibration
    mmc_boff[0]=getEfloat(4);
    mmc_boff[1]=getEfloat(8);
    mmc_boff[2]=getEfloat(12);
  }
  if((v[3]&0x02)!=0){ //load zero calibration
    mmc_soff[0]=getEfloat(16);
    mmc_soff[1]=getEfloat(20);
    mmc_soff[2]=getEfloat(24);
  }
}

void save_cal(byte type){
  byte v[4];
  
  v[0] = EEPROM.read(0);
  v[1] = EEPROM.read(1);
  v[2] = EEPROM.read(2);
  v[3] = EEPROM.read(3);
  if(v[0]!=0x11 || v[1]!=0x22 || v[2]!=0xA5){ //unique code to specify programming has occurred
    EEPROM.write(0, 0x11);
    EEPROM.write(1, 0x22);
    EEPROM.write(2, 0xA5);
    EEPROM.write(3, type);
    v[3]=type;
  }
  if((v[3]&type)==0){
    EEPROM.write(3, v[3]|type);
  }
  if((type&0x01)!=0){ //save bridge calibration
    putEfloat(4, mmc_boff[0]);
    putEfloat(8, mmc_boff[1]);
    putEfloat(12, mmc_boff[2]);
  }
  if((type&0x02)!=0){ //save zero calibration
    putEfloat(16, mmc_soff[0]);
    putEfloat(20, mmc_soff[1]);
    putEfloat(24, mmc_soff[2]);
  }
}

union{
    float fval;
    byte bval[4];
  } fconv;

float getEfloat(int adr){
  fconv.bval[0]=EEPROM.read(adr);
  fconv.bval[1]=EEPROM.read(adr+1);
  fconv.bval[2]=EEPROM.read(adr+2);
  fconv.bval[3]=EEPROM.read(adr+3);
  return fconv.fval;
}

void putEfloat(int adr, float val){
  adr=constrain(adr,0,127);
  fconv.fval = val;
  EEPROM.update(adr, fconv.bval[0]);
  EEPROM.update(adr+1, fconv.bval[1]);
  EEPROM.update(adr+2, fconv.bval[2]);
  EEPROM.update(adr+3, fconv.bval[3]);
}

void mag_startup(){
  digitalWrite(SSegEnablePin, HIGH); //turn on 7seg display 5V output
	delay(100);
	if(!ina232_check()){
		sflags[0]=symE;
		sflags[1]=iflags[1]; //error 1 : INA232 not found
		sflags[2]=0;
		sflags[3]=0;
		errorState=-1000; //error, lp indefinitely
		return;
	}
	set7sValue(ina_bus_voltage(), 2); //display voltage
	sflags[0]=symv; //v symbol: v5.12
	// check current and output it
  startup_millis=0;
  while(startup_millis<1500){
    //delay(1500); //wait for voltage to be visible
    show7sDigitTimeLimit();
  }
	set7sValue(ina_bus_current(), 2); //display current
	sflags[0]=symA; // A0.12
	//delay(1500); //wait for current to be visible
  startup_millis=0;
  while(startup_millis<1500){
    show7sDigitTimeLimit();
  }
  if(!mmc_check()){
    sflags[0]=symE;
    sflags[1]=iflags[2]; //error 2 : MMC not found
    sflags[2]=0;
    sflags[3]=0;
    errorState=-1000; //error, lp indefinitely
    return;
  }
	set7sValue(mmc_get_temp(), 1);
	sflags[0]=symt; // t25.1
	//delay(1500); //wait for temperature to be visible
  startup_millis=0;
  while(startup_millis<1500){
    show7sDigitTimeLimit();
  }
	/*sflags[0]=symb;
	sflags[1]=symC;
	sflags[2]=0;
	sflags[3]=0; // bC
	mmc_bridge_cal(20);*/
	if(digitalRead(PowerGoodPin)){
		digitalWrite(LEDPowerPin, HIGH);
		errorState=1000; //visualize with LEDs
	}else{
		//undervoltage error
    digitalWrite(LEDPowerPin, LOW);
    errorState=10; //don't visualize with LEDs
    sflags[0]=symu;
    sflags[1]=symv;
    sflags[2]=symE; //uvE = under voltage error
    sflags[3]=0;
    startup_millis=0;
    while(startup_millis<1500){//wait for message to be visible
      show7sDigitTimeLimit();
    }
	}
	return;
}

bool newEvent=false;
int freeCyclesCtr=0, freeCyclesLast=0;

void loop() {
  show7sDigitTimeLimit();
  touchBtnProcess(); //this causes issues with leds.show() pulse width if it runs while the signal is still being sent (1.5 ms after start)
  processSerial();
  
  //debugLEDs();
  
	if(errorState == -10){ //serial debug mode
		//processSerial();
		//debug7s();
		delay(2);
		return;
	}
	if(errorState == -1000){ //error mode
		//processSerial();
		delay(2);
		return;
	}
  
  if(touchBtnEvent[1]==1){ //right button pressed
    touchBtnEvent[1]=0; //clear event
    if(submenu7!=0){
      menu7=0;//return to default mode after an action
    }else{
      menu7++;
    }
    submenu7=0;
    newEvent=true;
  }
  if(touchBtnEvent[0]==1){ //left button pressed
    touchBtnEvent[0]=0; //clear event
    submenu7++;
    newEvent=true;
  }
  
	//common loop
	//acquire magnetic data and visualize
	if(mmc_get_mag_nice_timelimit()){
		switch(menu7){
      default:
      menu7=0;
			case 0: //field measurement
			set7sValue(mmc_out[3], 0); //write total field in [mG]
      if(newEvent){
        if(submenu7==1){
          visualizeMode++;
          submenu7=0;
        }
        leds.setBuffer(0);
      }
			break;
			case 1: //voltage,current,temperature
      switch(submenu7){
        default:
        submenu7=0;
        case 0:
        set7sValue(ina_bus_voltage(), 2); //display voltage
        sflags[0]=symv; //v symbol: v5.12
        break;
        case 1:
        set7sValue(ina_bus_current(), 2); //display current
        sflags[0]=symA; // A0.12
        break;
        case 2:
        set7sValue(mmc_get_temp(), 1); //display temperature
        sflags[0]=symt; // t25.1
        break;
      }
			break;
      case 2: //calibration
      sflags[0]=symC;
      sflags[1]=symA;
      sflags[2]=symL;
      switch(submenu7){
        default:
        submenu7=0;
        case 0:
        sflags[3]=0;
        break;
        case 1:
        sflags[3]=symb; //bridge calibration
        if(newEvent){
          startup_millis=0;
        }else{
          if(startup_millis > 4000){ //start after a delay
            mmc_bridge_cal(32,false);
            menu7=0; //return to operating mode
            submenu7=0;
          }
        }
        break;
        case 2:
        sflags[3]=iflags[0]; //zero calibration
        if(newEvent){
          startup_millis=0;
        }else{
          if(startup_millis > 4000){ //start after a delay
            mmc_bridge_cal(64,true);
            menu7=0; //return to operating mode
            submenu7=0;
          }
        }
        break;
        case 3: //write calibration to flash
        sflags[3]=symF;
        if(newEvent){
          startup_millis=0;
        }else{
          if(startup_millis > 4000){ //start after a delay
            save_cal(0x03); //write both bridge and zero cal values
            sflags[0]|=sdot;
            sflags[1]|=sdot;
            sflags[2]|=sdot;
            sflags[3]|=sdot;
            startup_millis=0;
            while(startup_millis<500){//wait for message to be visible
              show7sDigitTimeLimit();
            }
            menu7=0; //return to operating mode
            submenu7=0;
          }
        }
        break;
      }
      break;
      case 3: //flashlight with current readout
        if(submenu7<=0 || errorState<100){
          sflags[0]=symF;
          sflags[1]=symL;
          sflags[2]=iflags[5];
          sflags[3]=0;
        }else{
          if(newEvent){
            if(submenu7<9){
                for(int ii=0; ii<totalLEDs; ii++){
                  leds.setPixel(ii,(0x010101 << (submenu7-1)));
                }
              //leds.setBuffer(0x01<<(submenu7-1));
            }else{
              if(submenu7<12){
                for(int ii=0; ii<totalLEDs; ii++){
                  leds.setPixel(ii,(0xFF << (8*(submenu7-9))));
                }
              }else{
                submenu7=0;
                leds.setBuffer(0);
              }
            }
            leds.show();
            touchBtnAcqTime=0; //reset touch controller so it does not sample new buttons for a while, to avoid led.show() timing jitter
          }
          set7sValue(ina_bus_current(), 2); //display current
          sflags[0]=iflags[submenu7] | sdot; // #.0.12
        }
      break;
      case 4: //spherical harmonics
      if(newEvent){
        if(submenu7>numsphc || submenu7<0 || errorState<100){
          submenu7=0;
        }
        if(submenu7==0){
          sflags[0]=iflags[5];
          sflags[1]=symP;
          sflags[2]=symH;
          sflags[3]=0;
          leds.setBuffer(0);
          leds.show();
        }else{
          sflags[0]=iflags[sphci1[submenu7-1]];
          sflags[1]=0;
          sflags[2]=sdash;
          sflags[3]=iflags[sphci2[submenu7-1]];
          displaySphHCoeff(submenu7-1, d_alpha);
        }
        touchBtnAcqTime=0; //reset touch controller so it does not sample new buttons for a while, to avoid led.show() timing jitter
      }
      break;
		}

    show7sDigitTimeLimit();
  
  	if(errorState < 100 || menu7>2){
  		//LEDs powered off due to undervoltage, or also flashlight / spherical mode avoids visualization
      if(errorState < 100){
        //sflags[0]=symu;
        sflags[0]|=sdot; //set first decimal point
      }
  		//sflags[3]|=sdot; //set last decimal point
  	}else{
  		//visualize field with LEDs
  		//the mapping from MMC sensor to sphere is as follows:
  		//sensor Z- = sphere X+
  		//sensor X+ = sphere Y+
  		//sensor Y+ = sphere Z+
  		float x,y,z,g; //normalize field direction
  		x=-mmc_out[2]/mmc_out[3];
  		y=mmc_out[0]/mmc_out[3];
  		z=mmc_out[1]/mmc_out[3];
  		switch(visualizeMode){
        default:
        visualizeMode=0;
        brightMode++;
        switch(brightMode){
          default:
          brightMode=0;
          case 0:
          npb=0x0F;
          d_alpha=16;
          break;
          case 1:
          npb=0x40;
          d_alpha=44;
          break;
          case 2:
          npb=0xF0;
          d_alpha=128;
          break;
        }
        case 0: 
        //no dynamic visualization
        if(newEvent){
          leds.show();
        }
        break;
  			case 1:
  			displayNearestPoint(x,y,z,npb, 1);
  			break;
  			case 2:
  			displayNearestPoint(x,y,z,npb, 3);
  			break;
			case 3:
  			displaySphereGradient(x,y,z,d_alpha);
  			break;
        case 4:
        displaySphereHalo(x,y,z,d_alpha);
        break;
        case 5:
        //g=d_gamma*pow(mmc_out[3]/8000,d_theta); //scale overall brightness by total magnetic field
        //displaySphereGradient(x,y,z,g,d_beta);
        g=mmc_out[3]/1200;
        g=constrain(g,0,1);
        displayGrowingSphere(x, y, z, g, true, d_alpha);
        break;
        case 6:
        g=mmc_out[3]/1200;
        g=constrain(g,0,1);
        displayGrowingSphere(x, y, z, g, false, d_alpha);
        break;
  		}
      //delay(2); //wait for all the bits to be written out, to avoid timing jitter
      touchBtnAcqTime=0; //reset touch controller so it does not sample new buttons for a while, to avoid led.show() timing jitter
  	}

    newEvent=false;
	}else{
		freeCyclesCtr++;
	}
}

uint32_t readInt(){
  char c = Serial.read();
  uint32_t qr=0;
  while(c>='0' && c<='9'){
    qr=qr*10+(c-'0');
    c = Serial.read();
  }
  return qr;
}

void processSerial(){
	if(!Serial.available()){
		return;
	}
	char cin;
	int iin;
	switch(Serial.read()){
		case '?':
		Serial.println("c : check ICs");
		Serial.println("v : read voltage and current");
		Serial.println("5 | 0 : turn 5V bus on | off");
		Serial.println("t : read MMC temperature");
		Serial.println("m : read MMC magnetic field");
		Serial.println("b : read touch buttons");
    Serial.println("C : print MMC calibration");
    Serial.println("B : MMC bridge calibration");
    Serial.println("Z : MMC zero field calibration");
	Serial.println("f : display free cycles");
    Serial.println("w# : set MMC bandwidth, # = 0 to 3, 3 is fastest, 0 default");
    Serial.println("h : MMC histogram test, output 1000 values");
    Serial.println("r# : set alpha parameter, # = 0 to 255");
    Serial.println("q# : set beta parameter, # = 0 to 999 -> 0.00 to 9.99");
    Serial.println("X : illuminate sphere coord X+,Y+,Z+ points in R,G,B");
    Serial.println("F : flash all LEDs");
    Serial.println("S : magnetic SET current pulse");
    Serial.println("R : magnetic RESET current pulse");
    break;
		case 'c':
		if(ina232_check()){
			Serial.println("INA232 success");
		}else{
			Serial.println("INA232 fail");
		}
		if(mmc_check()){
			Serial.println("MMC success");
		}else{
			Serial.println("MMC fail");
		}
		break;
		case 'v':
		Serial.print("Voltage: ");
		Serial.println(ina_bus_voltage());
		Serial.print("Current: ");
		Serial.println(ina_bus_current());
		break;
		case '5':
		digitalWrite(LEDPowerPin, HIGH);
		Serial.println("5V LEDs On");
		break;
		case '0':
		digitalWrite(LEDPowerPin, LOW);
		Serial.println("5V LEDs Off");
		break;
		case 't':
    Serial.println("Meas time / us, T raw, T / degC");
		for(iin=0; iin<5; iin++){
      mmc_print_temp();
    }
		break;
		case 'm':
    Serial.println("Meas time / us, X, Y, Z raw");
		mmc_avarr[0]=0;
    mmc_avarr[1]=0;
    mmc_avarr[2]=0;
		for(iin=0; iin<mmc_niceavg; iin++){
      mmc_print_mag();
      mmc_avarr[0]+=mmc_in[0];
      mmc_avarr[1]+=mmc_in[1];
      mmc_avarr[2]+=mmc_in[2];
    }
    Serial.println("Averaged raw without offsets");
    Serial.print(mmc_avarr[0]/(1.0f*mmc_niceavg));
    Serial.print(",");
    Serial.print(mmc_avarr[1]/(1.0f*mmc_niceavg));
    Serial.print(",");
    Serial.println(mmc_avarr[2]/(1.0f*mmc_niceavg));
    mmc_avarr[0]=0; //prepare for next nice call
    mmc_avarr[1]=0;
    mmc_avarr[2]=0;
    mmc_conversion_active=1;
		break;
		case 'b':
		Serial.print("Touch button 1: ");
		Serial.println(touchBtnValue[0]); //values are updated in the background
		Serial.print("Touch button 2: ");
		Serial.println(touchBtnValue[1]);
		break;
    case 'C':
    Serial.println("Existing bridge offsets [mG] X,Y,Z");
    Serial.print(mmc_boff[0]);
    Serial.print(',');
    Serial.print(mmc_boff[1]);
    Serial.print(',');
    Serial.println(mmc_boff[2]);
    Serial.println("Existing zero offsets [mG] X,Y,Z");
    Serial.print(mmc_soff[0]);
    Serial.print(',');
    Serial.print(mmc_soff[1]);
    Serial.print(',');
    Serial.println(mmc_soff[2]);
    break;
    case 'B':
    mmc_bridge_cal(32,false);
    Serial.println("New bridge offsets [mG] X,Y,Z");
    Serial.print(mmc_boff[0]);
    Serial.print(',');
    Serial.print(mmc_boff[1]);
    Serial.print(',');
    Serial.println(mmc_boff[2]);
    break;
    case 'Z':
    mmc_bridge_cal(64,true);
    Serial.println("New bridge offsets [mG] X,Y,Z");
    Serial.print(mmc_boff[0]);
    Serial.print(',');
    Serial.print(mmc_boff[1]);
    Serial.print(',');
    Serial.println(mmc_boff[2]);
    Serial.println("New zero offsets [mG] X,Y,Z");
    Serial.print(mmc_soff[0]);
    Serial.print(',');
    Serial.print(mmc_soff[1]);
    Serial.print(',');
    Serial.println(mmc_soff[2]);
    break;
	case 'f':
	Serial.println(freeCyclesLast);
	break;
    case 'w':
    cin=Serial.read();
    iin=cin-'0';
    iin=constrain(iin,0,3);
    mmc_write(0x0A, (uint8_t)iin);
    Serial.print("Bandwidth set to ");
    Serial.println(iin);
    break;
    case 'h':
    for(iin=0; iin<1000; iin++){
      mmc_print_mag(); //this runs as fast as bandwidth allows
    }
    break;
    case 'r':
    iin=readInt();
    iin=constrain(iin,0,255);
    d_alpha=iin;
    break;
    case 'q':
    iin=readInt();
    iin=constrain(iin,0,999);
    d_beta=iin/100.0f;
    break;
    case 'X':
    leds.setBuffer(0);
    leds.setPixel(23,0x0F0000); //X+ red
    leds.setPixel(72,0x000F00); //Y+ green
    leds.setPixel(48,0x00000F); //Z+ blue
    leds.show();
    delay(1000);
    break;
    case 'F':
    Serial.println("Flash V,I=");
    leds.setBuffer(0xFF);
    leds.show();
    for(int i=0; i<7; i++){
      delay(34);
      Serial.print(ina_bus_voltage());
      Serial.print(", ");
      Serial.println(ina_bus_current());
    }
    leds.setBuffer(0);
    leds.show();
    break;
    case 'S':
    mmc_write(0x09, 0b00001000); //write register 09h Internal Control 0, SET current 500 ns
    Serial.println("SET complete");
    break;
    case 'R':
    mmc_write(0x09, 0b00010000); //write register 09h Internal Control 0, RESET current 500 ns
    Serial.println("RESET complete");
    break;
		default:
		Serial.println("Unknown command");
		break;
	}
	while(Serial.available()){
		Serial.read();
	}
}

void debug7s(){
	sflags[0]++;
	sflags[1]++;
	sflags[2]++;
	sflags[3]++;
}


elapsedMillis sseg_time;
//EventResponder sseg_update_evt; //7seg digit updates will take place during yield() calls
const int s7delay=2; //[ms] delay between updates

//somewhere in setup:
void set7setup(){
  for(int i=0; i<4; i++){
    pinMode(dpins[i],OUTPUT);
    digitalWriteFast(dpins[i],HIGH); //inverted
  }
  for(int i=0; i<8; i++){
    pinMode(spins[i],OUTPUT);
    digitalWriteFast(spins[i],LOW);
  }
  //sseg_update_evt.attach(&show7sDigitTimeLimit);
}

void set7sValue(float val, int decimal){
  bool ovr=false;
  bool r0=false;
  int a=0;
  //decimal=0 -> ___0 to 9999
  //decimal=1 -> __0.0 to 999.9
  //decimal=2 -> _0.00 to 99.99
  //decimal=3 -> 0.000 to 9.999
  //error -> xxxx.
  
  if(decimal<0 || decimal>3){
    decimal=0;
  }
  a=decimal;
  while(a>0){
    val=val*10;
    a--;
  }
  
  if(val<0){
    val=0;
    ovr=true;
  }
  if(val>=10000){
    val=9999;
    ovr=true;
  }
  a=floor(val/1000);
  if((a>0 && a<10)||(a==0 && decimal==3)){
    sflags[0]=iflags[a];
    val=val-a*1000;
    r0=true;
  }else{
    sflags[0]=0;
  }
  a=floor(val/100);
  if(r0 || (a>0 && a<10) ||(a==0 && decimal==2)){
    sflags[1]=iflags[a];
    val=val-a*100;
    r0=true;
  }else{
    sflags[1]=0;
  }
  a=floor(val/10);
  if(r0 || (a>0 && a<10)||(a==0 && decimal==1)){
    sflags[2]=iflags[a];
    val=val-a*10;
    r0=true;
  }else{
    sflags[2]=0;
  }
  a=floor(val);
  if(a>=0 && a<10){
    sflags[3]=iflags[a];
    val=val-a;
    r0=true;
  }else{
    sflags[3]=0;
  }
  if(ovr){
    sflags[3]|=128; //set last decimal point
  }
  if(decimal!=0){
    sflags[3-decimal]|=128; //set decimal point
  }
}

void show7sDigit(){
  //this should be called repeatedly
  digitalWriteFast(dpins[cdigit],HIGH); //turn off existing digit (inverted)
  cdigit=(cdigit+1)%4; //next digit
  byte j=1;
  for(int i=0; i<8; i++){
    digitalWriteFast(spins[i],(sflags[cdigit]&j)!=0); //turn off by writing LOW
    j<<=1;
  }
  digitalWriteFast(dpins[cdigit],LOW); //turn on new digit
}

void show7sDigitTimeLimit(){
  if(sseg_time<s7delay)
    return;
  show7sDigit();
  sseg_time=0;
}


float mmc_get_temp(){
  mmc_write(0x09, 0b00000010);//write to register 09h Internal Control 0
  //take temperature measurement
  startup_millis=0;
  while(startup_millis<mmc_delayms){//wait for message to be visible
    show7sDigitTimeLimit();
  }
  
  if(!mmc_read_bytes(mmc_buf, 0x07, 1)) //read temperature output byte
  {
  Serial.println("I2C error");
  }
  float temp=-75+0.8f*mmc_buf[0]; //[degC]
  return temp;
}

void mmc_write(byte reg, byte val){
  Wire.beginTransmission(MMC_addr);
  Wire.write(reg);
  Wire.write(val);
  Wire.endTransmission();
}

bool mmc_read_bytes(byte buf[], byte reg, byte num){
  Wire.beginTransmission(MMC_addr);
  Wire.write(reg); //read from register
  if(Wire.endTransmission(false)){
    Wire.endTransmission();
    return false;
  }
  int a=0;
  Wire.requestFrom(MMC_addr, num);
  while(num>0){
    buf[a]=Wire.read();
  a++;
  num--;
  }
  return true;
}

bool mmc_check(){
  if(!mmc_read_bytes(mmc_buf, 0x2F, 1)) //read from register 2Fh Product ID
  {
    Serial.println("I2C read fail");
    return false;
  }
  return mmc_buf[0]==0b00110000; //datasheet ID value
}

void mmc_start(){
  mmc_write(0x0A, 0b10000000); //write register 0Ah Internal Control 1, bit SW_RST for software reset, 8 ms measurement time
  //to switch to other bandwidth:
  //mmc_write(0x0A, 0b00000000); //8 ms measurement time, 100 Hz
  //mmc_write(0x0A, 0b00000001); //4 ms, 200 Hz
  //mmc_write(0x0A, 0b00000010); //2 ms, 400 Hz
  //mmc_write(0x0A, 0b00000011); //0.5 ms, 800 Hz
  delay(30); //wait for power-up
  //perform a SET + RESET to clear any previous magnetic field
  mmc_write(0x09, 0b00001000); //write register 09h Internal Control 0, SET current 500 ns
  delay(11);
  mmc_write(0x09, 0b00010000); //write register 09h Internal Control 0, RESET current 500 ns
  delay(1); 
  //Serial.println("software restart command sent");
}

bool mmc_get_mag_nice_60hz(){
  //we are running at higher bandwidth
  if(mmc_conversion_start < (mmc_delaymsfast * 1000)){
    return false; //data not ready yet
  }
  mmc_read_mag(); // +H + Offset
  mmc_intgarr[0]+=mmc_in[0]*cos_stamp; //X cos
  mmc_intgarr[3]+=mmc_in[0]*sin_stamp; //X sin
  mmc_intgarr[1]+=mmc_in[1]*cos_stamp;
  mmc_intgarr[4]+=mmc_in[1]*sin_stamp;
  mmc_intgarr[2]+=mmc_in[2]*cos_stamp;
  mmc_intgarr[5]+=mmc_in[2]*sin_stamp;
  mmc_conversion_active++;
  //start acquisition ahead of next call
  mmc_write(0x09, 0b00000001);//write to register 09h Internal Control 0 take magnetic measurement
  uint32_t ts=micros() % 50000; //60 Hz wave has 3 cycles in 50000 microseconds, 50 Hz wave has 1 cycle in 20000 microseconds
  mmc_conversion_start=0;
  float tsf = ts*(3.7699112e-4f);  //(3*(2*pi)/50000) gives range of 0 to 6*pi [rad] between 0 and 50000 microseconds
  cos_stamp=cos(tsf); //will be used in next call
  sin_stamp=sin(tsf);
  if(mmc_conversion_active<=mmc_intgavg){
    return false; //averaging not done
  }
  //at this point the mmc_intgarr has sum over mmc_intgavg datapoints of cos(phase)*mag and sin(phase)*mag
  mmc_intgarr[0]/=(16.0f*mmc_intgavg); //convert to [mG] ?
  mmc_intgarr[3]/=(16.0f*mmc_intgavg);
  mmc_out[0]=sqrt(mmc_intgarr[0]*mmc_intgarr[0]+mmc_intgarr[3]*mmc_intgarr[3]); //keep magnitude only
  mmc_intgarr[1]/=(16.0f*mmc_intgavg); //convert to [mG] ?
  mmc_intgarr[4]/=(16.0f*mmc_intgavg);
  mmc_out[1]=sqrt(mmc_intgarr[1]*mmc_intgarr[1]+mmc_intgarr[4]*mmc_intgarr[4]); //keep magnitude only
  mmc_intgarr[2]/=(16.0f*mmc_intgavg); //convert to [mG] ?
  mmc_intgarr[5]/=(16.0f*mmc_intgavg);
  mmc_out[2]=sqrt(mmc_intgarr[2]*mmc_intgarr[2]+mmc_intgarr[5]*mmc_intgarr[5]); //keep magnitude only
  //the magnitudes above are only positive, so should be visualized in a different manner from direct field
  mmc_out[3]=sqrt(mmc_out[0]*mmc_out[0]+mmc_out[1]*mmc_out[1]+mmc_out[2]*mmc_out[2]); //overall magnitude
  mmc_intgarr[0]=0; //prepare for next call
  mmc_intgarr[1]=0;
  mmc_intgarr[2]=0;
  mmc_intgarr[3]=0;
  mmc_intgarr[4]=0;
  mmc_intgarr[5]=0;
  mmc_conversion_active=1;
  return true; //data is ready
}

bool mmc_get_mag_nice_timelimit(){
  if(mmc_conversion_start < (mmc_delayms * 1000)){
    return false; //data not ready yet
  }
  mmc_read_mag(); // +H + Offset
  mmc_avarr[0]+=mmc_in[0];
    mmc_avarr[1]+=mmc_in[1];
    mmc_avarr[2]+=mmc_in[2];
  mmc_conversion_active++;
  //start acquisition ahead of next call
  mmc_write(0x09, 0b00000001);//write to register 09h Internal Control 0 take magnetic measurement
  mmc_conversion_start=0;
  freeCyclesLast=freeCyclesCtr;
  freeCyclesCtr=0;
  if(mmc_conversion_active<=mmc_niceavg){
    return false; //averaging not done
  }
  mmc_out[0]=mmc_avarr[0]/(16.0f*mmc_niceavg)+mmc_boff[0]+mmc_soff[0]; //convert to [mG]
  mmc_out[1]=mmc_avarr[1]/(16.0f*mmc_niceavg)+mmc_boff[1]+mmc_soff[1];
  mmc_out[2]=mmc_avarr[2]/(16.0f*mmc_niceavg)+mmc_boff[2]+mmc_soff[2];
  mmc_out[3]=sqrt(mmc_out[0]*mmc_out[0]+mmc_out[1]*mmc_out[1]+mmc_out[2]*mmc_out[2]);
  mmc_avarr[0]=0; //prepare for next call
  mmc_avarr[1]=0;
  mmc_avarr[2]=0;
  mmc_conversion_active=1;
  return true; //data is ready
}

bool mmc_read_mag(){
  if(!mmc_read_bytes(mmc_buf, 0x00, 7)) //read magnetic field output bytes
  {
    return false;
  }
  uint32_t xm = 0, ym = 0, zm = 0;
  xm=mmc_buf[0];
  xm<<=8;
  xm|=mmc_buf[1];
  xm<<=2;
  xm|=(mmc_buf[6]&0b11000000)>>6;
  ym=mmc_buf[2];
  ym<<=8;
  ym|=mmc_buf[3];
  ym<<=2;
  ym|=(mmc_buf[6]&0b00110000)>>4;
  zm=mmc_buf[4];
  zm<<=8;
  zm|=mmc_buf[5];
  zm<<=2;
  zm|=(mmc_buf[6]&0b00001100)>>2;
  mmc_in[0]=xm;
  mmc_in[1]=ym;
  mmc_in[2]=zm;
  return true;
}
/*
void mmc_read_mag_nice(int avg){
  avg=constrain(avg, 1, 16384);
  int a=avg;
  mmc_avarr[0]=0;
  mmc_avarr[1]=0;
  mmc_avarr[2]=0;
  while(a>0){
    mmc_start_mag();//take magnetic measurement
  delay(mmc_delayms);
    mmc_read_mag(); // +H + Offset
  mmc_avarr[0]+=mmc_in[0];
    mmc_avarr[1]+=mmc_in[1];
    mmc_avarr[2]+=mmc_in[2];
  a--;
  }
  mmc_out[0]=mmc_avarr[0]/(16.0f*avg)+mmc_boff[0]+mmc_soff[0]; //convert to [mG]
  mmc_out[1]=mmc_avarr[1]/(16.0f*avg)+mmc_boff[1]+mmc_soff[1];
  mmc_out[2]=mmc_avarr[2]/(16.0f*avg)+mmc_boff[2]+mmc_soff[2];
  mmc_out[3]=sqrt(mmc_out[0]*mmc_out[0]+mmc_out[1]*mmc_out[1]+mmc_out[2]*mmc_out[2]);
}
*/
void mmc_print_mag(){
  mmc_write(0x09, 0b00000001);//write to register 09h Internal Control 0
  //take magnetic measurement
  mmc_conversion_start=0;
  bool meas=true;
  int stall=0;
  while(meas){
  mmc_read_bytes(mmc_buf, 0x08, 1);
  meas=(mmc_buf[0]&0x01)==0; //mmc_buf will show 0x01 when magnetic measurement is finished
  stall++;
  if(stall > 10000){
    Serial.println("MMC stall error");
    return;
  }
  }
  uint32_t ts=mmc_conversion_start;
  //Serial.print("MMC magnetic measurement time / us = ");
  Serial.print(ts);
  Serial.print(',');
  if(!mmc_read_mag()) //read magnetic field output bytes
  {
  Serial.println("Data read error");
  }
  
  //Serial.print("Mag X=");
  Serial.print(mmc_in[0]);
  //Serial.print("Mag Y=");
  Serial.print(',');
  Serial.print(mmc_in[1]);
  //Serial.print("Mag Z=");
  Serial.print(',');
  Serial.println(mmc_in[2]);
}

void mmc_print_temp(){
  mmc_write(0x09, 0b00000010);//write to register 09h Internal Control 0
  mmc_conversion_start=0;
  //take temperature measurement
  bool meas=true;
  int stall=0;
  while(meas){
  mmc_read_bytes(mmc_buf, 0x08, 1);
  meas=(mmc_buf[0]&0x02)==0; //mmc_buf will show 0x02 when temperature measurement is finished
  stall++;
  if(stall > 10000){
    Serial.println("MMC stall error");
    return;
  }
  }
  uint32_t ts=mmc_conversion_start;
  //Serial.print("MMC temperature measurement time / us = ");
  Serial.print(ts);
  Serial.print(',');
  if(!mmc_read_bytes(mmc_buf, 0x07, 1)) //read temperature output byte
  {
  Serial.println("I2C error");
  }
  //Serial.print("T=");
  Serial.print(mmc_buf[0]);
  Serial.print(',');
  Serial.println(-75+0.8f*mmc_buf[0]);
}

void mmc_bridge_cal(int avg, bool soffzero){
  avg=constrain(avg, 1, 8192);
  sflags[0]=symb;
  sflags[1]=symC;
  sflags[2]=iflags[5];
  sflags[3]=0;
  mmc_write(0x09, 0b00001000); //write register 09h Internal Control 0, SET current 500 ns
  int a=avg;
  mmc_avarr[0]=0;
  mmc_avarr[1]=0;
  mmc_avarr[2]=0;
  mmc_svarr[0]=0;
  mmc_svarr[1]=0;
  mmc_svarr[2]=0;
  while(a>0){
    mmc_write(0x09, 0b00000001);//start magnetic measurement
    //delay(mmc_delayms);
    startup_millis=0;
    while(startup_millis<mmc_delayms){
      show7sDigitTimeLimit();
    }
    mmc_read_mag(); // +H + Offset
    mmc_avarr[0]+=mmc_in[0];
    mmc_avarr[1]+=mmc_in[1];
    mmc_avarr[2]+=mmc_in[2];
    mmc_svarr[0]+=mmc_in[0];
    mmc_svarr[1]+=mmc_in[1];
    mmc_svarr[2]+=mmc_in[2];
  a--;
  }
  /*Serial.print(mmc_avarr[0]);
  Serial.print(',');
  Serial.print(mmc_avarr[1]);
  Serial.print(',');
  Serial.println(mmc_avarr[2]);*/
  sflags[2]=symr;
  mmc_write(0x09, 0b00010000); //write register 09h Internal Control 0, RESET current 500 ns
  a=avg;
  while(a>0){
    mmc_write(0x09, 0b00000001);//start magnetic measurement
    //delay(mmc_delayms);
    startup_millis=0;
    while(startup_millis<mmc_delayms){
      show7sDigitTimeLimit();
    }
    mmc_read_mag(); // -H + Offset
    mmc_avarr[0]+=mmc_in[0];
    mmc_avarr[1]+=mmc_in[1];
    mmc_avarr[2]+=mmc_in[2];
    mmc_svarr[0]-=mmc_in[0];
    mmc_svarr[1]-=mmc_in[1];
    mmc_svarr[2]-=mmc_in[2];
  a--;
  }
  //avarr = (+H + Offset -H + Offset)*avg = (2*Offset)*avg
  //svarr = (+H + Offset +H -Offset)*avg = (2*H)*avg
  mmc_boff[0]=mmc_avarr[0]/(avg*(16*-2.0f)); //calculated bridge offset as [mG]
  mmc_boff[1]=mmc_avarr[1]/(avg*(16*-2.0f));
  mmc_boff[2]=mmc_avarr[2]/(avg*(16*-2.0f));
  if(soffzero){
    mmc_soff[0]=mmc_svarr[0]/(avg*(16*2.0f)); //calculated absolute zero offset as [mG]
    mmc_soff[1]=mmc_svarr[1]/(avg*(16*2.0f));
    mmc_soff[2]=mmc_svarr[2]/(avg*(16*2.0f));
  }
  mmc_avarr[0]=0; //prepare for next nice call
  mmc_avarr[1]=0;
  mmc_avarr[2]=0;
  mmc_conversion_active=1;
}

const byte ina_addr=0x49; //0x48 or 0x49
const float isense_r=0.03f; //sense resistor resistance [ohm]

uint16_t i2c_read16(byte addr, byte reg){
  Wire1.beginTransmission(addr);
  Wire1.write(reg);
  if(Wire1.endTransmission(false)){
    Wire1.endTransmission();
    Serial.println("I2C 16bit read fail");
    return 0;
  }
  Wire1.requestFrom(addr, 2);
  uint16_t ret=Wire1.read();
  ret<<=8;
  ret|=Wire1.read();
  return ret;
}

bool i2c_write16(byte addr, byte reg, uint16_t val){
  Wire1.beginTransmission(addr);
  Wire1.write(reg);
  byte val1;
  val1=(val>>8)&0xFF;
  Wire1.write(val1);
  val1=val&0xFF;
  Wire1.write(val1);
  bool r=(Wire1.endTransmission() == 0);
  delay(1);
  return r;
}

bool ina232_check(){
  return (i2c_read16(ina_addr, 0x3E)==0x5449); //check the Manufacturer ID register
}

bool ina232_start(){
  //send a reset command
  i2c_write16(ina_addr, 0x00, 0xC000);
  delay(30);
  //return i2c_write16(ina_addr, 0x00, 0x49B7);//set configuration for 128 averages, 4156 us conversion time (data ready every 532 ms), continuous mode
  return i2c_write16(ina_addr, 0x00, 0x456F);//set configuration for 16 averages, 2116 us conversion time (data ready every 33 ms), continuous mode
}

float ina_shunt_voltage(){
  int vs = i2c_read16(ina_addr, 0x01);
  if(vs>=0x8000){
    vs-=0x10000; //convert to signed
  }
  return vs * 2.5e-6f;
}

float ina_bus_voltage(){
  int vs = i2c_read16(ina_addr, 0x02);
  return vs * 1.6e-3f;
}

float ina_bus_current(){
  return ina_shunt_voltage()/isense_r;
}


//EventResponder tbtn_update_evt; //button updates will take place during yield() calls

void touchBtnSetup(){
  //set up touch buttons
  for(uint_fast8_t btn=0; btn<numTouchBtns; btn++){
    *portConfigRegister(touchBtnPin[btn]) = PORT_PCR_MUX(0);
    // touchBtnTSI[btn]=pin2tsi[touchBtnPin[btn]];
  }
  
  //set up TSI module
  noInterrupts();
  SIM_SCGC5 |= SIM_SCGC5_TSI; //turn on TSI
  interrupts();

  //TSI_GENCS_REFCHRG = [0.5, 1, 2, 4, 8, 16, 32, 64] uA charge reference cap (1 pF)
  //TSI_GENCS_EXTCHRG = [0.5, 1, 2, 4, 8, 16, 32, 64] uA charge external cap
  //TSI_GENCS_DVOLT = [0.3, 0.45, 0.6, 0.67] V difference on oscillator
  //TSI_GENCS_PS = [1, 2, 4, 8, 16, 32, 64, 128] external oscillator frequency divider
  //TSI_GENCS_NSCN = 1-32 scans per electrode (higher count value = longer time to count up reference)

  delay(10);
  
  //## Note: TSI_GENCS_ESOR causes error so apparently nothing happens? conversion always returns 1
  TSI0_GENCS = TSI_GENCS_REFCHRG(4)|TSI_GENCS_EXTCHRG(6)|TSI_GENCS_PS(2)|TSI_GENCS_NSCN(2)|TSI_GENCS_TSIEN;
    //ref charge 8uA, ext charge 32 uA
    //divide by 4 and ncount 3
    //enable TSI
  
  //tbtn_update_evt.attach(&touchBtnProcess);
}

void touchBtnProcess(){
  if(touchBtnAcqState==0){
    touchBtnAcqTime=0;
    TSI0_DATA = TSI_DATA_TSICH(touchBtnTSI[touchBtnI]) | TSI_DATA_SWTS; //this starts the TSI capacitance measurement - about 10 microseconds
    touchBtnAcqState=1;
  }else{
    if(touchBtnAcqTime < touchBtnDelay)
      return;
    touchBtnValue[touchBtnI]=TSI0_DATA & 0xFFFF;
    if(touchBtnPressed[0] || touchBtnPressed[1]){//allow only one at a time due to cross-talk
      if(touchBtnPressed[touchBtnI] && (touchBtnValue[touchBtnI] < (touchBtnThres-touchBtnHyst))){
        touchBtnPressed[touchBtnI]=false;
        touchBtnEvent[touchBtnI]=-1; //released
      }
    }else{
      if((touchBtnValue[touchBtnI] > touchBtnThres) && (touchBtnValue[touchBtnI] > touchBtnValue[1-touchBtnI])){
        touchBtnPressed[touchBtnI]=true;
        touchBtnEvent[touchBtnI]=1; //pressed
      }
    }
    touchBtnI=(touchBtnI+1)%numTouchBtns;
    touchBtnAcqState=0;
  }
}
