﻿using System;
using static System.Math;

namespace AstroDisplayUI
{
    class SolarSystemCalc
    {
        //based on code from http://stjarnhimlen.se/comp/tutorial.html by Paul Schlyter pausch@stjarnhimlen.se

        //Ecliptic geocentric position (_Ecl): C1=longitude from vernal equinox, C2=latitude from ecliptic plane
        //Ecliptic heliocentric position (_Hel): Cs as above
        //Equatorial geocentric position (_Eq): C1=right ascension from vernal equinox, C2=declination from equator
        //Azimuthal geo(topo)centric position (_Az): C1=azimuth from north, C2=altitude from horizon
        internal AstroPosition SunEcl, SunEq, EarthHel, SunAz;
        internal AstroPosition MoonEcl, MoonEq, MoonHel, MoonAz; //note all Moon positions other than MoonHel are in earth-moon length scale
        internal AstroPosition MercuryEcl, MercuryEq, MercuryHel, MercuryAz;
        internal AstroPosition VenusEcl, VenusEq, VenusHel, VenusAz;
        internal AstroPosition MarsEcl, MarsEq, MarsHel, MarsAz;
        internal AstroPosition JupiterEcl, JupiterEq, JupiterHel, JupiterAz;
        internal AstroPosition SaturnEcl, SaturnEq, SaturnHel, SaturnAz;
        internal AstroPosition UranusEcl, UranusEq, UranusHel, UranusAz;
        internal AstroPosition NeptuneEcl, NeptuneEq, NeptuneHel, NeptuneAz;
        internal double MoonIllumination, MoonPhase;//0 = no illumination, 1 = full illumination; phase=0 at full illumination, pi at no illumination
        internal double MercuryMag, VenusMag, MarsMag, JupiterMag, SaturnMag, UranusMag, NeptuneMag, MoonMag; //lower value means brighter object

        public const double DTOR = PI / 180;

        public void Calculate(DateTime dobs, double obs_latitude_degn, double obs_longitude_dege)
        {
            //dobs = local date and time of observation
            //obs_latitude_degn = observation latitude in degrees north (-90 to 90, 39 for Washington DC), for azimuth and altitude
            //obs_longitude_dege = observation longitude in degrees east (-180 to 180, 77 for Washington DC), for approximate UTC correction, not too important

            DateTime dobs_utc = dobs - TimeSpan.FromDays(obs_longitude_dege / 360);//approximate UTC date of observation
            //the above conversion does not take into account time zones or equation of time
            //since this is only used to calculate orbit parameters which change slowly
            //there is a DateTime.ToUniversalTime() function but it uses an implicit time zone
            DateTime d0 = new DateTime(1999, 12, 31);
            //day number
            TimeSpan dayNum = dobs_utc - d0;
            double d = dayNum.TotalDays; //[days] since Dec 31 1999 UST (approximate) used for orbital parameters

            double ecl = (23.4393 - (3.563e-7 * d)%360) * DTOR;//[rad] obliquity of the ecliptic

            OrbitalElements Sun = new OrbitalElements(new double[] { 0, 0, 0, 0, 282.9404, 4.70935e-5, 1, 0, 0.016709, -1.151e-9, 356.0470, 0.9856002585 }, d);
            //compute position of the sun
            SunEcl = GetEcliptic(Sun);
            double[] sunpos = SunEcl.GetRectangular();
            EarthHel = new AstroPosition() { C1 = SunEcl.C1 + PI, C2 = -SunEcl.C2, Radius = SunEcl.Radius };
            SunEq = AstroPosition.EclRotate(SunEcl, ecl);

            //convert further to local coordinates
            double obs_latitude = obs_latitude_degn * DTOR; //[rad]
            double obs_localtime = dobs.TimeOfDay.TotalHours * (PI / 12); //[rad] 0 to 2*pi, pi = noon
            //the above local time does not take into account time zones or equation of time
            //therefore the result may correspond to almanac entries up to ~30 minutes earlier or later than the time specified in dobs
            double Ls = Sun.w + Sun.M; //mean longitude
            double gmst0 = Ls + PI;
            double sidtime = obs_localtime + gmst0;
            SunAz = AstroPosition.HorRotate(AstroPosition.ToHourAngle(SunEq, sidtime), PI / 2 - obs_latitude);

            OrbitalElements Moon = new OrbitalElements(new double[] {125.1228, -0.0529538083, 5.1454, 0, 318.0634, 0.1643573223, 60.2666, 0, 0.0549, 0, 115.3654, 13.0649929509 }, d);

            MoonEcl = GetEcliptic(Moon);
            //Moon perturbations
            double Lm = Moon.N + Moon.w + Moon.M;
            double D = Lm - Ls;
            double F = Moon.w + Moon.M;
            double plon =
                -1.274 * Sin(Moon.M - 2 * D)
                + 0.658 * Sin(2 * D)
                - 0.186 * Sin(Sun.M)
                - 0.059 * Sin(2 * Moon.M - 2 * D)
                - 0.057 * Sin(Moon.M - 2 * D + Sun.M)
                + 0.053 * Sin(Moon.M + 2 * D)
                + 0.046 * Sin(2 * D - Sun.M)
                + 0.041 * Sin(Moon.M - Sun.M)
                - 0.035 * Sin(D)
                - 0.031 * Sin(Moon.M + Sun.M)
                - 0.015 * Sin(2 * F - 2 * D)
                + 0.011 * Sin(Moon.M - 4 * D);
            double plat =
                -0.173 * Sin(F - 2 * D)
                - 0.055 * Sin(Moon.M - F - 2 * D)
                - 0.046 * Sin(Moon.M + F - 2 * D)
                + 0.033 * Sin(F + 2 * D)
                + 0.017 * Sin(2 * Moon.M + F);
            double pdis =
                -0.58 * Cos(Moon.M - 2 * D)
                - 0.46 * Cos(2 * D);
            MoonEcl.C1 += plon * DTOR;
            MoonEcl.C2 += plat * DTOR;
            MoonEcl.Radius += pdis;
            double[] moonpos = MoonEcl.GetRectangular();
            double aur = 4.26343e-5; //[au / earth equatorial radius]
            MoonHel = new AstroPosition(moonpos[0] * aur - sunpos[0], moonpos[1] * aur - sunpos[1], moonpos[2] * aur - sunpos[2]);
            MoonEq = AstroPosition.EclRotate(MoonEcl, ecl);
            MoonAz = AstroPosition.HorRotate(AstroPosition.ToHourAngle(MoonEq, sidtime), PI / 2 - obs_latitude);
            //topocentric correction (simple form)
            double mpar = Asin(1 / MoonAz.Radius);
            MoonAz.C2 = MoonAz.C2 - mpar * Cos(MoonAz.C2);
            //moon illumination
            double elong = Acos(Cos(SunEcl.C1 - MoonEcl.C1) * Cos(MoonEcl.C2));
            MoonPhase = PI - elong;
            MoonIllumination = (1 + Cos(MoonPhase)) / 2;
            double moonr = MoonAz.Radius / 23450; //moon radius in au
            MoonMag = 0.23 + 5 * Log10(moonr * SunEcl.Radius) + 0.026 * (MoonPhase / DTOR) + 4e-9 * Pow(MoonPhase / DTOR, 4);

            OrbitalElements Mercury = new OrbitalElements(new double[] { 48.3313, 3.24587e-5, 7.0047, 5.00e-8, 29.1241, 1.01444e-5, 0.387098, 0, 0.205635, 5.59e-10, 168.6562, 4.0923344368 }, d);
            OrbitalElements Venus = new OrbitalElements(new double[] { 76.6799, 2.46590e-5, 3.3946, 2.75e-8, 54.8910, 1.38374e-5, 0.723330, 0, 0.006773, -1.302e-9, 48.0052, 1.6021302244 }, d);
            OrbitalElements Mars = new OrbitalElements(new double[] { 49.5574, 2.11081e-5, 1.8497, -1.78e-8, 286.5016, 2.92961e-5, 1.523688, 0, 0.093405, 2.516e-9, 18.6021, 0.5240207766 }, d);
            OrbitalElements Jupiter = new OrbitalElements(new double[] { 100.4542, 2.76854e-5, 1.3030, -1.557e-7, 273.8777, 1.64505e-5, 5.20256, 0, 0.048498, 4.469e-9, 19.8950, 0.0830853001 }, d);
            OrbitalElements Saturn = new OrbitalElements(new double[] { 113.6634, 2.38980e-5, 2.4886, -1.081e-7, 339.3939, 2.97661e-5, 9.55475, 0, 0.055546, -9.499e-9, 316.9670, 0.0334442282 }, d);
            OrbitalElements Uranus = new OrbitalElements(new double[] { 74.0005, 1.3978e-5, 0.7733, 1.9e-8, 96.6612, 3.0565e-5, 19.18171, -1.55e-8, 0.047318, 7.45e-9, 142.5905, 0.011725806 }, d);
            OrbitalElements Neptune = new OrbitalElements(new double[] { 131.7806, 3.0173e-5, 1.7700, -2.55e-7, 272.8461, -6.027e-6, 30.05826, 3.313e-8, 0.008606, 2.15e-9, 260.2471, 0.005995147 }, d);

            MercuryHel = GetEcliptic(Mercury);
            MercuryEcl = MakeGeocentric(MercuryHel, sunpos);
            MercuryEq = AstroPosition.EclRotate(MercuryEcl, ecl);
            MercuryAz = AstroPosition.HorRotate(AstroPosition.ToHourAngle(MercuryEq, sidtime), PI / 2 - obs_latitude);
            FV fv = GetFV(MercuryEcl, MercuryHel, SunEcl);
            MercuryMag = -0.36 + fv.Item2 + 0.027 * fv.Item1 + 2.2e-13 * Pow(fv.Item1, 6);

            VenusHel = GetEcliptic(Venus);
            VenusEcl = MakeGeocentric(VenusHel, sunpos);
            VenusEq = AstroPosition.EclRotate(VenusEcl, ecl);
            VenusAz = AstroPosition.HorRotate(AstroPosition.ToHourAngle(VenusEq, sidtime), PI / 2 - obs_latitude);
            fv = GetFV(VenusEcl, VenusHel, SunEcl);
            VenusMag = -4.34 + fv.Item2 + 0.013 * fv.Item1 + 4.2e-7 * Pow(fv.Item1, 3);

            MarsHel = GetEcliptic(Mars);
            MarsEcl = MakeGeocentric(MarsHel, sunpos);
            MarsEq = AstroPosition.EclRotate(MarsEcl, ecl);
            MarsAz = AstroPosition.HorRotate(AstroPosition.ToHourAngle(MarsEq, sidtime), PI / 2 - obs_latitude);
            fv = GetFV(MarsEcl, MarsHel, SunEcl);
            MarsMag = -1.51 + fv.Item2 + 0.016 * fv.Item1;

            JupiterHel = GetEcliptic(Jupiter);
            //perturbations
            double plonj =
                - 0.332 * Sin(2 * Jupiter.M - 5 * Saturn.M - 67.6 * DTOR)
                - 0.056 * Sin(2 * Jupiter.M - 2 * Saturn.M + 21 * DTOR)
                + 0.042 * Sin(3 * Jupiter.M - 5 * Saturn.M + 21 * DTOR)
                - 0.036 * Sin(Jupiter.M - 2 * Saturn.M)
                + 0.022 * Cos(Jupiter.M - Saturn.M)
                + 0.023 * Sin(2 * Jupiter.M - 3 * Saturn.M + 52 * DTOR)
                - 0.016 * Sin(Jupiter.M - 5 * Saturn.M - 69 * DTOR);
            JupiterHel.C1 += plonj * DTOR;
            JupiterEcl = MakeGeocentric(JupiterHel, sunpos);
            JupiterEq = AstroPosition.EclRotate(JupiterEcl, ecl);
            JupiterAz = AstroPosition.HorRotate(AstroPosition.ToHourAngle(JupiterEq, sidtime), PI / 2 - obs_latitude);
            fv = GetFV(JupiterEcl, JupiterHel, SunEcl);
            JupiterMag = -9.25 + fv.Item2 + 0.014 * fv.Item1;

            SaturnHel = GetEcliptic(Saturn);
            //perturbations
            double plons =
                0.812 * Sin(2 * Jupiter.M - 5 * Saturn.M - 67.6 * DTOR)
                - 0.229 * Cos(2 * Jupiter.M - 4 * Saturn.M - 2 * DTOR)
                + 0.119 * Sin(Jupiter.M - 2 * Saturn.M - 3 * DTOR)
                + 0.046 * Sin(2 * Jupiter.M - 6 * Saturn.M - 69 * DTOR)
                + 0.014 * Sin(Jupiter.M - 3 * Saturn.M + 32 * DTOR);
            double plats =
                -0.020 * Cos(2 * Jupiter.M - 4 * Saturn.M - 2 * DTOR)
                + 0.018 * Sin(2 * Jupiter.M - 6 * Saturn.M - 49 * DTOR);
            SaturnHel.C1 += plons * DTOR;
            SaturnHel.C2 += plats * DTOR;
            SaturnEcl = MakeGeocentric(SaturnHel, sunpos);
            SaturnEq = AstroPosition.EclRotate(SaturnEcl, ecl);
            SaturnAz = AstroPosition.HorRotate(AstroPosition.ToHourAngle(SaturnEq, sidtime), PI / 2 - obs_latitude);
            fv = GetFV(SaturnEcl, SaturnHel, SunEcl);
            double ir = 28.06 * DTOR;
            double Nr = ((169.51 + 3.82e-5 * d) % 360) * DTOR;
            double B = Asin(Sin(SaturnEcl.C2) * Cos(ir) - Cos(SaturnEcl.C2) * Sin(ir) * Sin(SaturnEcl.C1 - Nr));
            double ring_magn = -2.6 * Sin(Abs(B)) + 1.2 * Pow(Sin(B), 2);
            SaturnMag = -9.0 + fv.Item2 + 0.044 * fv.Item1 + ring_magn;

            UranusHel = GetEcliptic(Uranus);
            //perturbations
            double plonu =
                0.040 * Sin(Saturn.M - 2 * Uranus.M + 6 * DTOR)
                + 0.035 * Sin(Saturn.M - 3 * Uranus.M + 33 * DTOR)
                - 0.015 * Sin(Jupiter.M - Uranus.M + 20 * DTOR);
            UranusHel.C1 += plonu * DTOR;
            UranusEcl = MakeGeocentric(UranusHel, sunpos);
            UranusEq = AstroPosition.EclRotate(UranusEcl, ecl);
            UranusAz = AstroPosition.HorRotate(AstroPosition.ToHourAngle(UranusEq, sidtime), PI / 2 - obs_latitude);
            fv = GetFV(UranusEcl, UranusHel, SunEcl);
            UranusMag = -7.15 + fv.Item2 + 0.001 * fv.Item1;

            NeptuneHel = GetEcliptic(Neptune);
            NeptuneEcl = MakeGeocentric(NeptuneHel, sunpos);
            NeptuneEq = AstroPosition.EclRotate(NeptuneEcl, ecl);
            NeptuneAz = AstroPosition.HorRotate(AstroPosition.ToHourAngle(NeptuneEq, sidtime), PI / 2 - obs_latitude);
            fv = GetFV(NeptuneEcl, NeptuneHel, SunEcl);
            NeptuneMag = -6.90 + fv.Item2 + 0.001 * fv.Item1;
        }

        private AstroPosition GetEcliptic(OrbitalElements orb)
        {
            //mean and true anomaly
            double E = orb.M + orb.e * Sin(orb.M) * (1 + orb.e * Cos(orb.M));
            E = E - (E - orb.M - orb.e * Sin(E)) / (1 - orb.e * Cos(E));//can iterate multiple times
            
            double xv = orb.a * (Cos(E) - orb.e);
            double yv = orb.a * Sqrt(1 - (orb.e * orb.e)) * Sin(E);
            
            AstroPosition ap = new AstroPosition(xv, yv, 0); //coordinates in local orbit
            //Radius = distance, C1 = true anomaly, C2 = 0
            //rotate to ecliptic
            double vw = ap.C1 + orb.w;
            double x = ap.Radius * (Cos(orb.N) * Cos(vw) - Sin(orb.N) * Sin(vw) * Cos(orb.i));
            double y = ap.Radius * (Sin(orb.N) * Cos(vw) + Cos(orb.N) * Sin(vw) * Cos(orb.i));
            double z = ap.Radius * Sin(vw) * Sin(orb.i);
            return new AstroPosition(x, y, z);
        }

        private AstroPosition MakeGeocentric(AstroPosition helio, double[] sun)
        {
            double[] rc = helio.GetRectangular();
            return new AstroPosition(rc[0] + sun[0], rc[1] + sun[1], rc[2] + sun[2]);
        }

        private struct FV
        {
            public double Item1, Item2;
            public FV(double i1, double i2)
            {
                Item1 = i1;
                Item2 = i2;
            }
        }

        private FV GetFV(AstroPosition planetGeo, AstroPosition planetHelio, AstroPosition sunGeo)
        {
            return new FV(
                Acos((planetHelio.Radius * planetHelio.Radius + planetGeo.Radius * planetGeo.Radius - sunGeo.Radius * sunGeo.Radius) / (2 * planetHelio.Radius * planetGeo.Radius)) / DTOR, 
                5 * Log10(planetGeo.Radius * planetHelio.Radius)
                );
        }

        internal class OrbitalElements
        {
            internal double N, i, w, a, e, M;

            public OrbitalElements() { }
            public OrbitalElements(double[] o, double d)
            {
                //o (size 12) contains offset and shift in degrees for N, i, w, a, e, M in that order
                if (o.Length != 12)
                    throw new ArgumentException("Incorrect number of parameters");
                N = (o[0] + (o[1] * d) % 360) * DTOR;
                i = (o[2] + (o[3] * d) % 360) * DTOR;
                w = (o[4] + (o[5] * d) % 360) * DTOR;
                a = o[6] + (o[7] * d);
                e = o[8] + (o[9] * d);
                M = (o[10] + (o[11] * d) % 360) * DTOR;
            }
        }


        internal class AstroPosition
        {
            public double Radius, C1, C2;
            //public enum Pos { Equatorial, Ecliptic, Azimuth };
            //Equatorial: C1=RA, C2=Decl [rad]
            //Ecliptic: C1=Longitude, C2=Latitude [rad]
            //Azimuth: C1=Asimuth, C2=Altitude [rad]
            //public Pos Type;

            public double[] GetRectangular()
            {
                double[] v = new double[3];
                v[0] = Radius * Cos(C1) * Cos(C2);
                v[1] = Radius * Sin(C1) * Cos(C2);
                v[2] = Radius * Sin(C2);
                return v;
            }

            public AstroPosition() { }
            public AstroPosition(double x, double y, double z)
            {
                Radius = Sqrt(x * x + y * y + z * z);
                double pd = Sqrt(x * x + y * y);
                C1 = Atan2(y, x);
                C2 = Atan2(z, pd);
            }

            public static AstroPosition EclRotate(AstroPosition p1, double ecl)
            {
                double[] sr = p1.GetRectangular();
                double x = sr[0];
                double y = sr[1] * Cos(ecl) - sr[2] * Sin(ecl);
                double z = sr[1] * Sin(ecl) + sr[2] * Cos(ecl);
                return new AstroPosition(x, y, z);
            }

            public static AstroPosition HorRotate(AstroPosition p1, double hr)
            {
                double[] sr = p1.GetRectangular();
                double x = sr[0] * Cos(hr) - sr[2] * Sin(hr);
                double y = sr[1];
                double z = sr[0] * Sin(hr) + sr[2] * Cos(hr);
                AstroPosition r = new AstroPosition(x, y, z);
                r.C1 += PI; //measure azimuth from north
                return r;
            }

            public static AstroPosition ToHourAngle(AstroPosition p1, double sidtime)
            {
                return new AstroPosition() { C1 = sidtime - p1.C1, C2 = p1.C2, Radius = p1.Radius };
            }

            public override string ToString()
            {
                return "{" + (C1 / DTOR).ToString() + " deg, " + (C2 / DTOR).ToString() + " deg, " + Radius.ToString() + "}";
            }
        }

    }
}
