﻿using Microsoft.Win32;
using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Windows;
using System.Windows.Media;

namespace AstroDisplayUI
{
    /// <summary>
    /// Interaction logic for MainWindow.xaml
    /// </summary>
    public partial class MainWindow : Window
    {
        //double Lat = 39 + 8.0 / 60, Lon = -77 - 12.0 / 60; //Gaithersburg
        //double Lat = 45.5097, Lon = 6.7059; //Belle Plagne

        StarsCalc stc;
        SolarSystemCalc ssc;
        LaserProjector lpj;
        bool ready = false;//prevent early valuechanged events

        public MainWindow()
        {
            InitializeComponent();
            lpj = new LaserProjector();
            ssc = new SolarSystemCalc();
            SkyPlot.SetColors(Brushes.DarkBlue, Brushes.LightGray);
            GndPlot.SetColors(Brushes.DarkSlateGray, Brushes.LightGray);
            db.Text = DateTime.Now.ToString();
            HShift.Value = DateTime.Now.TimeOfDay.TotalDays * 359.9;
            Button_Click(null, null);
            stc = new StarsCalc();
            stc.ParseStarsFile();
            stc.ParseConstellationsFile();
            ConstellationsList.Items.Clear();
            ConstellationsList.ItemsSource = stc.Constellations.Keys;
            ready = true;
            Button_Click(null, null);//start first calculation
        }

        private void Button_Click(object sender, RoutedEventArgs e)
        {
            if (!ready || stc == null || ssc == null)
                return;

            double Lat = double.Parse(Latitude.Text);
            double Lon = double.Parse(Longitude.Text);

            DateTime dobs = DateTime.Parse(db.Text); //time is referenced to local noon - longitude and equation of time not considered
            db.Text = dobs.ToString(); //update formatting
            if (lpj.IsConnected && HoldDisplay.IsChecked != true)
                lpj.ClearBuffer();
            EclPlot.ClearPlot();
            SkyPlot.ClearPlot();
            GndPlot.ClearPlot();
            PlotConstellations(dobs, Lat, Lon);//plot in order of brightness
            PlotStars(dobs, Lat, Lon);
            SSCalc(dobs, Lat, Lon);
            if (Orbit24t.IsChecked == true)
            {
                for (int dh = -24; dh <= 24; dh++)
                {
                    if (dh != 0)
                    {
                        SSCalc(dobs.AddHours(dh/2.0), Lat, Lon);
                    }
                }
            }
            if (Orbit24es.IsChecked == true)
            {
                DateTime[] eqs = { new DateTime(2025, 5, 1), new DateTime(2025, 8, 1), new DateTime(2025, 11, 1), new DateTime(2025, 2, 1) };
                for (int et = 0; et < 4; et++)
                {
                    DateTime de = eqs[et];
                    for (int dh = -12; dh <= 12; dh++)
                    {
                        SSCalc(de.AddHours(dh), Lat, Lon);
                    }
                }
            }
            if (Orbitmd.IsChecked == true)
            {
                for (int dh = -15; dh <= 15; dh++)
                {
                    if (dh != 0)
                    {
                        SSCalc(dobs.AddDays(dh), Lat, Lon);
                    }
                }
            }
            if (Orbityw.IsChecked == true)
            {
                for (int dh = -25; dh <= 25; dh++)
                {
                    if (dh != 0)
                    {
                        SSCalc(dobs.AddDays(7 * dh), Lat, Lon);
                    }
                }
            }
            if (lpj.IsConnected)
                lpj.SendData();
        }

        private void SSCalc(DateTime dobs, double lat, double lon)
        {
            //SolarSystemCalc ssc = new SolarSystemCalc();
            //ssc = new SolarSystemCalc();
            //Test case presented at stjarhimlen.se
            //ssc.Calculate(new DateTime(1990, 4, 19, 1, 0, 0), 60, 15);

            ssc.Calculate(dobs, lat, lon);

            tb.Text = "The sun is at " + (ssc.SunAz.C1 / Math.PI * 180).ToString() + " deg azimuth and " + (ssc.SunAz.C2 / Math.PI * 180).ToString() + " deg altitude";
            tm.Text = "The moon is at " + (ssc.MoonAz.C1 / SolarSystemCalc.DTOR).ToString() + " deg azimuth and " + (ssc.MoonAz.C2 / SolarSystemCalc.DTOR).ToString() + " deg altitude";

            //EclPlot.ClearPlot();
            Brush f = Brushes.DarkRed;
            double farsc = 10;
            EclPlot.PlotPol(0, 0, f, 5, "Sun"); //Sun
            EclPlot.PlotPol(ssc.MercuryHel.C1, ssc.MercuryHel.Radius / farsc, f, 1, "Mercury"); //Mercury
            EclPlot.PlotPol(ssc.VenusHel.C1, ssc.VenusHel.Radius / farsc, f, 1, "Venus"); //Venus
            EclPlot.PlotPol(ssc.EarthHel.C1, ssc.EarthHel.Radius / farsc, Brushes.DarkBlue, 1, "Earth"); //Earth
            EclPlot.PlotPol(ssc.MoonHel.C1, ssc.MoonHel.Radius / farsc, Brushes.DarkBlue, 0.5, "Moon"); //Moon
            EclPlot.PlotPol(ssc.MarsHel.C1, ssc.MarsHel.Radius / farsc, f, 2, "Mars"); //Mars
            EclPlot.PlotPol(ssc.JupiterHel.C1, ssc.JupiterHel.Radius / farsc, f, 3, "Jupiter"); //Jupiter
            EclPlot.PlotPol(ssc.SaturnHel.C1, ssc.SaturnHel.Radius / farsc, f, 5, "Saturn"); //Saturn

            //SkyPlot.ClearPlot();
            //GndPlot.ClearPlot();
            if (SolarList.SelectedItems.Contains(SolarList.Items[0]))
                PlotAz(ssc.SunAz, "Sun", 3);
            if (SolarList.SelectedItems.Contains(SolarList.Items[1]))
                PlotAz(ssc.MercuryAz, "Mercury", 3);
            if (SolarList.SelectedItems.Contains(SolarList.Items[2]))
                PlotAz(ssc.VenusAz, "Venus", 3);
            if (SolarList.SelectedItems.Contains(SolarList.Items[3]))
                PlotAz(ssc.MoonAz, "Moon", 3);
            if (SolarList.SelectedItems.Contains(SolarList.Items[4]))
                PlotAz(ssc.MarsAz, "Mars", 3);
            if (SolarList.SelectedItems.Contains(SolarList.Items[5]))
                PlotAz(ssc.JupiterAz, "Jupiter", 3);
            if (SolarList.SelectedItems.Contains(SolarList.Items[6]))
                PlotAz(ssc.SaturnAz, "Saturn", 3);
            if (SolarList.SelectedItems.Contains(SolarList.Items[7]))
                PlotAz(ssc.UranusAz, "Uranus", 3);
            if (SolarList.SelectedItems.Contains(SolarList.Items[8]))
                PlotAz(ssc.NeptuneAz, "Neptune", 3);
        }

        private void PlotStars(DateTime dobs, double lat, double lon)
        {
            double maglimit = MagLimit.Value;
            IEnumerable<KeyValuePair<int, StarsCalc.StarInfo>> slc = stc.Stars.Where(p => p.Value.mag < maglimit);
            foreach (KeyValuePair<int, StarsCalc.StarInfo> kvp in slc)
            {
                StarsCalc.StarInfo sf = kvp.Value;
                SolarSystemCalc.AstroPosition sap = stc.CalculateStarPosition(sf, dobs, lat, lon);
                byte brt = (byte)(3 - Math.Round((sf.mag / maglimit) * 2));
                if (brt > 3) brt = 3;
                if (brt < 1) brt = 1;
                PlotAz(sap, sf.ToString(), brt);
            }
        }

        private void PlotConstellations(DateTime dobs, double lat, double lon)
        {
            SortedSet<int> cfs = new SortedSet<int>();
            double clml = CLMagLimit.Value;
            byte brt = (byte)(OutlineCS.IsChecked == true ? 3 : 2);
            foreach (string k in ConstellationsList.SelectedItems)
            {
                //StarsCalc.ConstellationFigure? cf = null; // stc?.Constellations.Where(p => p.Key.StartsWith("draco", StringComparison.OrdinalIgnoreCase)).First().Value;
                StarsCalc.ConstellationFigure cf = stc.Constellations[k];
                if (cf != null)
                {
                    foreach (StarsCalc.ConstellationFigure.CFLine si in cf.lines)
                    {
                        StarsCalc.StarInfo sf = stc.Stars[si.S1];
                        SolarSystemCalc.AstroPosition sap1 = stc.CalculateStarPosition(sf, dobs, lat, lon);
                        StarsCalc.StarInfo sf2 = stc.Stars[si.S2];
                        SolarSystemCalc.AstroPosition sap2 = stc.CalculateStarPosition(sf2, dobs, lat, lon);
                        if (DrawCL.IsChecked == true)
                        {
                            PlotAzLine(sap1, sap2);
                        }
                        if (cfs.Add(si.S1) && sf.mag < clml)
                        {
                            PlotAz(sap1, sf.ToString(), brt);
                        }
                        if (cfs.Add(si.S2) && sf2.mag < clml)
                        {
                            PlotAz(sap2, sf2.ToString(), brt);
                        }
                    }
                    cfs.Clear();
                }
            }
        }

        private void ToPresent_Click(object sender, RoutedEventArgs e)
        {
            DateTime dobs = DateTime.Now;
            db.Text = dobs.ToString();
            Button_Click(null, null);
        }

        private void HourFwd_Click(object sender, RoutedEventArgs e)
        {
            DateTime dobs = DateTime.Parse(db.Text).Add(GetTimeUnit());
            db.Text = dobs.ToString();
            Button_Click(null, null);
        }

        private TimeSpan GetTimeUnit()
        {
            if (tuyr.IsChecked == true)
                return TimeSpan.FromDays(356.25);
            if (tumn.IsChecked == true)
                return TimeSpan.FromDays(30);
            if (tudy.IsChecked == true)
                return TimeSpan.FromDays(1);
            //if (tuhr.IsChecked == true)
            return TimeSpan.FromHours(1);
        }

        private void HourBack_Click(object sender, RoutedEventArgs e)
        {
            DateTime dobs = DateTime.Parse(db.Text).Subtract(GetTimeUnit());
            db.Text = dobs.ToString();
            Button_Click(null, null);
        }

        private void PlotAz(SolarSystemCalc.AstroPosition az, string nm, byte brt=2)
        {
            if (az.C2 >= 0)
            {
                //visible in sky
                //C1 is measured from north to east, C2 from horizon up to zenith
                //north is at bottom of projected circle, east at right, so that standing to face north then raising the circle up it matches the sky
                SkyPlot.PlotPrj(az.C1 - Math.PI / 2, az.C2, brt==2?Brushes.LightGoldenrodYellow:Brushes.LightYellow, 2, nm);
                if (lpj.IsConnected)
                {
                    lpj.DrawPoint(-az.C1, Math.PI / 2 - az.C2, brt);//for projector, theta is from north to west, phi from zenith down to horizon
                }
            }
            else
            {
                //below horizon
                //C1 is measured from north to east, C2 from horizon up to zenith
                //north is at top of projected circle, east at right, so that standing to face north then lowering circle it matches seeing through the ground
                GndPlot.PlotPrj(Math.PI / 2 - az.C1, az.C2, brt==2?Brushes.OrangeRed:Brushes.Orange, 2, nm);
            }
        }

        private async void Connect_Projector_Click(object sender, RoutedEventArgs e)
        {
            if (!lpj.IsConnected)
            {

                Connect_Projector_Btn.Content = "Connecting";
                Connect_Projector_Btn.IsEnabled = false;
                await lpj.Connect();
                Connect_Projector_Btn.Content = "Connected";
                Connect_Projector_Btn.IsEnabled = true;
            }
            else
            {
                lpj.Disconnect();
                Connect_Projector_Btn.Content = "Connect to projector";
            }
        }

        private async void Update_Projector_Click(object sender, RoutedEventArgs e)
        {
            if (lpj.IsConnected)
            {
                lpj.Disconnect();
            }
            OpenFileDialog ofd = new OpenFileDialog();
            ofd.Filter = "HEX Files|*.hex";
            if (ofd.ShowDialog() == true)
            {
                Load_Firmware_Btn.Content = "Transferring file";
                Load_Firmware_Btn.IsEnabled = false;
                await lpj.UpdateFlash(ofd.FileName, (uint)ofd.OpenFile().Length);
                Load_Firmware_Btn.Content = "Load firmware";
                Load_Firmware_Btn.IsEnabled = true;
            }            
        }

        private void Speed_Click(object sender, RoutedEventArgs e)
        {
            if (lpj.IsConnected)
            {
                //double spd = lpj.GetMotorSpeed();
                //MessageBox.Show("Motor speed: " + spd.ToString() + " RPS");
                lpj.DrawGrid();
                lpj.SendData();
            }
        }

        private void PlotAzLine(SolarSystemCalc.AstroPosition az1, SolarSystemCalc.AstroPosition az2)
        {
            if (az1.C2 >= 0)
            {
                //visible in sky
                //C1 is measured from north to east
                //north is at bottom of projected circle, east at right, so that standing to face north then raising the circle up it matches the sky
                SkyPlot.PlotPrjLine(az1.C1 - Math.PI / 2, az1.C2, az2.C1 - Math.PI / 2, az2.C2, Brushes.LightGoldenrodYellow, 0.5);
                if (lpj.IsConnected)
                {
                    lpj.DrawLine((float)(-az1.C1), (float)(Math.PI / 2 - az1.C2), (float)(-az2.C1), (float)(Math.PI / 2 - az2.C2), 1);
                }
            }
            else
            {
                //below horizon
                //C1 is measured from north to east
                //north is at top of projected circle, east at right, so that standing to face north then lowering circle it matches seeing through the ground
                GndPlot.PlotPrjLine(Math.PI / 2 - az1.C1, az1.C2, Math.PI / 2 - az2.C1, az2.C2, Brushes.OrangeRed, 0.5);
            }
        }

        private void VShift_ValueChanged(object sender, RoutedPropertyChangedEventArgs<double> e)
        {
            if (!ready)
                return;
            Latitude.Text = VShift.Value.ToString("F1");
            Button_Click(null, null);
        }

        private void HShift_ValueChanged(object sender, RoutedPropertyChangedEventArgs<double> e)
        {
            if (!ready)
                return;
            DateTime dobs = DateTime.Parse(db.Text); //time is referenced to local noon - longitude and equation of time not considered
            DateTime dobss = dobs.Date + TimeSpan.FromDays(HShift.Value / 360.0);
            db.Text = dobss.ToString(); //update formatting
            Button_Click(null, null);
        }

        private void ConstellationsList_SelectionChanged(object sender, System.Windows.Controls.SelectionChangedEventArgs e)
        {
            Button_Click(null, null);
        }

        private void CSelectAll_Click(object sender, RoutedEventArgs e)
        {
            ConstellationsList.SelectAll();
        }

        private void CDeselectAll_Click(object sender, RoutedEventArgs e)
        {
            ConstellationsList.UnselectAll();
        }
        
        private void MagLimit_ValueChanged(object sender, RoutedPropertyChangedEventArgs<double> e)
        {
            Button_Click(null, null);
        }

        private void BrtMode_Checked(object sender, RoutedEventArgs e)
        {
            if (!ready) return;
            if (lpj.IsConnected)
            {
                lpj.SetBrightMode((byte)(BrtMode.IsChecked == true ? (BrtMode2.IsChecked == true ? 2 : 1) : 0));
            }
        }
    }
}